/*
 * cleopatre/application/p1905_managerd/src/cmdu_message_parse.c
 *
 * (C) Copyright 2013 MSsar Semiconductor, Inc.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <errno.h>
#include <linux/if_packet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include "cmdu_message_parse.h"
#include "cmdu_message.h"
#include "cmdu_fragment.h"
#include "cmdu_retry_message.h"
#include "multi_ap.h"
#include "cmdu.h"
#include "_1905_lib_io.h"
#include "byteorder.h"
#include "debug.h"
#include "os.h"
#include "topology.h"
#include "eloop.h"
#include "ethernet_layer.h"

extern int find_lldp_by_port_id_mac(unsigned char *receiving_mac, unsigned char *port_id_mac);

unsigned char rx_buffer[10240]={0};

extern const unsigned char p1905_multicast_address[6];

void topology_discovery_action(struct p1905_managerd_ctx *ctx, unsigned char *peer_al_mac,
	unsigned char need_query, unsigned char need_notify)
{
	int j = 0;

	/*if receive topology discovery message in new device, query it*/
	/* send topology query when get topology discovery */
	/* new add, 2013.11.21
	 * if UPDATE_BRIDGE_FDB_ENABLE = 1, no need to send query to neighbor
	 * each time when we get topology discovery to maintain FDBs in Linux.
	 * bridge.
	 */
	if (need_query) {
		 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
			   * instead of AL mac address
			   */
		debug(DEBUG_TRACE, "send topology query to %02x:%02x:%02x:%02x:%02x:%02x for discovery\n",
			PRINT_MAC(peer_al_mac));
		insert_cmdu_txq(peer_al_mac, ctx->p1905_al_mac_addr, e_topology_query, ++ctx->mid,
			ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
	}
	/*if local p1905.1 neighbor device info updated, notify*/
	if (need_notify == 1) {
		debug(DEBUG_TRACE, "send notification from %02x:%02x:%02x:%02x:%02x:%02x for discovery\n",
			PRINT_MAC(ctx->p1905_al_mac_addr));
		ctx->mid++;
		for (j = 0; j < ctx->itf_number; j++) {
			/*send multicast cmdu to all the interface */
			insert_cmdu_txq((unsigned char*)p1905_multicast_address, ctx->p1905_al_mac_addr,
				e_topology_notification, ctx->mid, ctx->itf[j].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, j);
#endif
		}
	} else if (need_notify == 2) {
		debug(DEBUG_TRACE, "send notification with mtk vendor tlv from"
			" %02x:%02x:%02x:%02x:%02x:%02x for discovery\n",
			PRINT_MAC(ctx->p1905_al_mac_addr));
		ctx->mid++;
		fill_send_tlv(ctx, peer_al_mac, ETH_ALEN);
		for (j = 0; j < ctx->itf_number; j++) {
			/*send multicast cmdu to all the interface */
			insert_cmdu_txq((unsigned char*)p1905_multicast_address,
				ctx->p1905_al_mac_addr,
				e_topology_notification_with_vendor_ie, ctx->mid, ctx->itf[j].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification_with_vendor_ie, ctx->mid, j);
#endif
		}
	}
}
/**
 *  add/update 1905.1 neighbor device information.
 *
 * \param  ctx  1905.1 managerd context
 * \param  al_mac  neighbor al mac addr in topology discovery message.
 * \param  neighbor_itf_mac  neighbor device interface mac in topology discover.
 * \param  itf_mac_addr  the interface topology discovery received from.
 * \param  notify  if need to send topology notify, set this flag
 * \param  retun error code
 */
int update_p1905_neighbor_dev_info(struct p1905_managerd_ctx *ctx,
    unsigned char *al_mac, unsigned char *neighbor_itf_mac,
    unsigned char *itf_mac_addr, unsigned char *notify)
{
    int i = 0;
    struct p1905_neighbor_info *dev_info = NULL;

    for (i=0; i<ctx->itf_number; i++) {
        if (!memcmp(ctx->p1905_neighbor_dev[i].local_mac_addr, itf_mac_addr, ETH_ALEN))
            break;
    }

	if (i >= ctx->itf_number) {
		debug(DEBUG_ERROR, "no this local interface(%s)\n", itf_mac_addr);
		return -1;
	}

    if (!LIST_EMPTY(&ctx->p1905_neighbor_dev[i].p1905nbr_head)) {
        LIST_FOREACH(dev_info, &ctx->p1905_neighbor_dev[i].p1905nbr_head,
                     p1905nbr_entry)
        {
            if (!memcmp(dev_info->al_mac_addr, al_mac, ETH_ALEN)) {
                if (dev_info->ieee802_1_bridge == 0) {
                    /* can not find any matched mac address in LLDP storage
                     * it means that local device just receive the topology
                     * discovery but no 802.1 bridge detection message.
                     * so a 802.1 bridge exist
                     */
                    if (0 == find_lldp_by_port_id_mac(itf_mac_addr, neighbor_itf_mac)) {
                        dev_info->ieee802_1_bridge = 1;
                        *notify = 1;
                    }
                }
                return 0;
            }
        }
    }

    dev_info = (struct p1905_neighbor_info *)malloc(sizeof(struct p1905_neighbor_info));
	if (!dev_info) {
		debug(DEBUG_ERROR, "alloc dev_info fail\n");
		return -1;
	}
    memcpy(dev_info->al_mac_addr, al_mac, ETH_ALEN);
    dev_info->ieee802_1_bridge = 1;
    LIST_INSERT_HEAD(&ctx->p1905_neighbor_dev[i].p1905nbr_head, dev_info, p1905nbr_entry);
	debug(DEBUG_OFF, RED("local_mac_addr(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
		PRINT_MAC(itf_mac_addr));
    /*need to send topology notification message*/
    *notify = 1;
    return 0;
}

/**
 *  delete 1905.1 topology response database.
 *
 * \param  ctx  1905.1 managerd context
 * \param  al_mac  input the target device al mac to delete.
 */
void delete_exist_topology_response_database(struct p1905_managerd_ctx *ctx,
                                                unsigned char *al_mac)
{
    struct topology_response_db *tpgr_db;
    struct device_info_db *dev_info, *dev_info_temp;
    struct p1905_neighbor_device_db *p1905_neighbor, *p1905_neighbor_temp;
    struct non_p1905_neighbor_device_list_db *non_p1905_neighbor, *non_p1905_neighbor_temp;
    struct device_bridge_capability_db *br_cap, *br_cap_temp;

    if(!SLIST_EMPTY(&(ctx->topology_entry.tprdb_head)))
    {
        SLIST_FOREACH(tpgr_db, &(ctx->topology_entry.tprdb_head), tprdb_entry)
        {
            if(!memcmp(tpgr_db->al_mac_addr, al_mac, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(tpgr_db->devinfo_head)))
                {
                    dev_info = SLIST_FIRST(&(tpgr_db->devinfo_head));
                    while(dev_info)
                    {
                        dev_info_temp = SLIST_NEXT(dev_info, devinfo_entry);
                        if(!SLIST_EMPTY(&(dev_info->p1905_nbrdb_head)))
                        {
                            /*delete all p1905.1 device in this interface*/
                            p1905_neighbor =\
                            SLIST_FIRST(&(dev_info->p1905_nbrdb_head));
                            while(p1905_neighbor)
                            {
                                p1905_neighbor_temp =\
                                SLIST_NEXT(p1905_neighbor, p1905_nbrdb_entry);

                                SLIST_REMOVE(&(dev_info->p1905_nbrdb_head),\
                                p1905_neighbor, p1905_neighbor_device_db,\
                                p1905_nbrdb_entry);
                                free(p1905_neighbor);
                                p1905_neighbor = p1905_neighbor_temp;
                            }
                        }

                        /*delete all non-p1905.1 device in this interface*/
                        if(!SLIST_EMPTY(&(dev_info->non_p1905_nbrdb_head)))
                        {
                            non_p1905_neighbor =\
                            SLIST_FIRST(&(dev_info->non_p1905_nbrdb_head));
                            while(non_p1905_neighbor)
                            {
                                non_p1905_neighbor_temp =\
                                SLIST_NEXT(non_p1905_neighbor, non_p1905_nbrdb_entry);

                                SLIST_REMOVE(&(dev_info->non_p1905_nbrdb_head),\
                                non_p1905_neighbor, non_p1905_neighbor_device_list_db,\
                                non_p1905_nbrdb_entry);
                                free(non_p1905_neighbor);
                                non_p1905_neighbor = non_p1905_neighbor_temp;
                            }
                        }
                        /*remove this device info db from list*/
                        SLIST_REMOVE(&(tpgr_db->devinfo_head), dev_info,
                                    device_info_db, devinfo_entry);
                        /*delete this device info db*/
                        if(dev_info->vs_info_len)
                            free(dev_info->vs_info);

                        free(dev_info);
                        dev_info = dev_info_temp;
                    }
                }

                /*delete bridge capability info*/
                if(!LIST_EMPTY(&(tpgr_db->brcap_head)))
                {
                    br_cap = LIST_FIRST(&(tpgr_db->brcap_head));
                    while(br_cap)
                    {
                        br_cap_temp = LIST_NEXT(br_cap, brcap_entry);
                        LIST_REMOVE(br_cap, brcap_entry);

                        if(br_cap->interface_amount)
                            free(br_cap->interface_mac_tuple);

                        free(br_cap);
                        br_cap = br_cap_temp;
                    }
                }
                /*finall, delete whole topology response with this AL MAC*/
                SLIST_REMOVE(&(ctx->topology_entry.tprdb_head), tpgr_db,\
                            topology_response_db, tprdb_entry);
                free(tpgr_db);
                return;
            }
        }
    }
}

struct topology_response_db * delete_exist_topology_response_content(
	struct p1905_managerd_ctx *ctx, unsigned char *al_mac)
{
	struct topology_response_db *tpgr_db = NULL;
	struct device_info_db *dev_info, *dev_info_temp;
	struct p1905_neighbor_device_db *p1905_neighbor, *p1905_neighbor_temp;
	struct non_p1905_neighbor_device_list_db *non_p1905_neighbor, *non_p1905_neighbor_temp;
	struct device_bridge_capability_db *br_cap, *br_cap_temp;

	SLIST_FOREACH(tpgr_db, &(ctx->topology_entry.tprdb_head), tprdb_entry) {
		if (os_memcmp(tpgr_db->al_mac_addr, al_mac, ETH_ALEN))
			continue;

		dev_info = SLIST_FIRST(&tpgr_db->devinfo_head);
		while (dev_info) {
			dev_info_temp = SLIST_NEXT(dev_info, devinfo_entry);
			/*delete all p1905.1 device in this interface*/
			p1905_neighbor = SLIST_FIRST(&dev_info->p1905_nbrdb_head);
			while (p1905_neighbor) {
				p1905_neighbor_temp = SLIST_NEXT(p1905_neighbor, p1905_nbrdb_entry);
				SLIST_REMOVE(&dev_info->p1905_nbrdb_head,
					p1905_neighbor, p1905_neighbor_device_db, p1905_nbrdb_entry);
				free(p1905_neighbor);
				p1905_neighbor = p1905_neighbor_temp;
			}
			/*delete all non-p1905.1 device in this interface*/
			non_p1905_neighbor = SLIST_FIRST(&dev_info->non_p1905_nbrdb_head);
			while (non_p1905_neighbor) {
				non_p1905_neighbor_temp =
					SLIST_NEXT(non_p1905_neighbor, non_p1905_nbrdb_entry);
				SLIST_REMOVE(&dev_info->non_p1905_nbrdb_head,
					non_p1905_neighbor, non_p1905_neighbor_device_list_db,
					non_p1905_nbrdb_entry);
				free(non_p1905_neighbor);
				non_p1905_neighbor = non_p1905_neighbor_temp;
			}
			/*remove this device info db from list*/
			SLIST_REMOVE(&tpgr_db->devinfo_head, dev_info, device_info_db,
				devinfo_entry);
			/*delete this device info db*/
			if(dev_info->vs_info_len)
				free(dev_info->vs_info);

			free(dev_info);
			dev_info = dev_info_temp;
		}

		/*delete bridge capability info*/
		br_cap = LIST_FIRST(&tpgr_db->brcap_head);
		while (br_cap) {
			br_cap_temp = LIST_NEXT(br_cap, brcap_entry);
			LIST_REMOVE(br_cap, brcap_entry);

			if(br_cap->interface_amount)
				free(br_cap->interface_mac_tuple);

			free(br_cap);
			br_cap = br_cap_temp;
		}
		return tpgr_db;
	}

	return tpgr_db;
}


void delete_exist_topology_discovery_database(struct p1905_managerd_ctx *ctx,
	unsigned char *al_mac, unsigned char *itf_mac)
{
	struct topology_discovery_db *tpg_db = NULL;

	if (!LIST_EMPTY(&ctx->topology_entry.tpddb_head)) {
        LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
        {
			if (!memcmp(tpg_db->al_mac, al_mac, 6) &&
				!memcmp(tpg_db->itf_mac, itf_mac, 6)) {
				delete_p1905_neighbor_dev_info(ctx, tpg_db->al_mac,
					(unsigned char *)tpg_db->receive_itf_mac);
				/*delete the topology discovery database*/
				LIST_REMOVE(tpg_db, tpddb_entry);
				free(tpg_db);
				return;
			}
        }
	}
}


/*
Fn to delete the topology rsp db by port by local interface mac address
	port == -1 delete by local interface mac address
	local_mac == NULL delete by port number index
*/
void delete_topo_response_db_by_port_interface(struct p1905_managerd_ctx *ctx,
	int port)
{
    struct topology_response_db *tpgr_db = NULL, *tpgr_db_tmp = NULL;
	unsigned char del_cnt = 0;

	if (port == -1) {
		debug(DEBUG_ERROR, "error port(-1)!!!\n");
		return;
	}

	tpgr_db = SLIST_FIRST(&ctx->topology_entry.tprdb_head);
	while (tpgr_db) {
		tpgr_db_tmp = SLIST_NEXT(tpgr_db, tprdb_entry);
		if (tpgr_db->eth_port == port) {
			delete_exist_topology_response_database(ctx, tpgr_db->al_mac_addr);
			del_cnt++;
		}
		tpgr_db = tpgr_db_tmp;
	}

	if (del_cnt)
		mark_valid_topo_rsp_node(ctx);
}

/*
Fn to delete the discovery db by port number or local interface mac address
	port == -1 delete by local interface mac address
	local_mac == NULL delete by port number index
*/
int delete_topo_disc_db_by_port(struct p1905_managerd_ctx *ctx,
	int port, unsigned char *exclude_almac)
{
	struct topology_discovery_db *db = NULL, *db_temp = NULL;
	int cnt = 0, del_rsp_cnt = 0;

	if (port == -1)
		return cnt;

	if (!LIST_EMPTY(&ctx->topology_entry.tpddb_head)) {
		db = LIST_FIRST(&ctx->topology_entry.tpddb_head);
		while (db) {
			db_temp = LIST_NEXT(db, tpddb_entry);
			if (db->eth_port == port)  {
				if (exclude_almac && !os_memcmp(exclude_almac, db->al_mac, ETH_ALEN)) {
					db = db_temp;
					continue;
				}
				delete_p1905_neighbor_dev_info(ctx, db->al_mac, db->receive_itf_mac);
				LIST_REMOVE(db, tpddb_entry);
				cnt++;
				debug(DEBUG_OFF, GRN("delete topology discovery db ALMAC(%02x:%02x:%02x:%02x:%02x:%02x)\n"
					"IFMAC(%02x:%02x:%02x:%02x:%02x:%02x) RECV_IFMAC(%02x:%02x:%02x:%02x:%02x:%02x) PORT[%d]"),
					PRINT_MAC(db->al_mac), PRINT_MAC(db->itf_mac), PRINT_MAC(db->receive_itf_mac),
					db->eth_port);
				if (find_discovery_by_almac(ctx, db->al_mac) == NULL) {
					debug(DEBUG_OFF, "delete topology rsp (%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(db->al_mac));
					delete_exist_topology_response_database(ctx, db->al_mac);
					del_rsp_cnt++;
				}
				os_free(db);
			}
			db = db_temp;
		}
	}

	if (del_rsp_cnt)
		mark_valid_topo_rsp_node(ctx);

	return cnt;
}


/**
 *  delete one 1905.1 neighbor device.
 *
 * \param  ctx  1905.1 managerd context
 * \param  al_mac  input the neighbor device al mac to delete.
 * \param  retun error code
 */
int delete_p1905_neighbor_dev_info(struct p1905_managerd_ctx *ctx,
	unsigned char *al_mac, unsigned char *recvif_mac)
{
    int i = 0;
    struct p1905_neighbor_info *dev_info;

    for (i=0; i<ctx->itf_number; i++) {
		if (!memcmp(recvif_mac, ctx->itf[i].mac_addr, ETH_ALEN)) {
            LIST_FOREACH(dev_info, &ctx->p1905_neighbor_dev[i].p1905nbr_head, p1905nbr_entry) {
                if (!memcmp(dev_info->al_mac_addr, al_mac, ETH_ALEN)) {
					debug(DEBUG_OFF, GRN("local inf(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
						PRINT_MAC(recvif_mac));
                    LIST_REMOVE(dev_info, p1905nbr_entry);
                    free(dev_info);
                    return 1;
                }
			}
		}
	}

	return 0;
}

struct topology_discovery_db *get_tpd_db_by_mac(struct p1905_managerd_ctx *ctx, unsigned char *mac)
{
	struct topology_discovery_db *tpg_db = NULL;

	LIST_FOREACH(tpg_db, &(ctx->topology_entry.tpddb_head), tpddb_entry) {
		if (!os_memcmp(tpg_db->itf_mac, mac, ETH_ALEN)) {
			break;
		}
	}

	return tpg_db;
}

struct topology_response_db *get_trsp_db_by_mac(struct p1905_managerd_ctx *ctx, unsigned char *mac)
{
	struct topology_response_db *trsp_db = NULL;

	SLIST_FOREACH(trsp_db, &(ctx->topology_entry.tprdb_head), tprdb_entry) {
		if (!os_memcmp(trsp_db->al_mac_addr, mac, ETH_ALEN))
			break;
	}

	return trsp_db;
}


/**
 *  update the time to live for topology dicovery message.
 *  if value of time to live is less than 0, need to delete this record
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  sec  for 1905.1 managerd use.
 * \return error code
 */
int delete_not_exist_p1905_neighbor_device(struct p1905_managerd_ctx *ctx,
                                            int sec)
{
    struct topology_discovery_db *tpg_db, *tpg_db_temp;
    int result = 0;
	char del_rsp_cnt = 0;

    if(!LIST_EMPTY(&(ctx->topology_entry.tpddb_head)))
    {
        tpg_db = LIST_FIRST(&(ctx->topology_entry.tpddb_head));
        while(tpg_db)
        {
            tpg_db_temp = LIST_NEXT(tpg_db, tpddb_entry);
            tpg_db->time_to_live -= sec;
            if(tpg_db->time_to_live <= 0)
            {
                /*delete the local p1905.1 neighbor device info*/
				debug(DEBUG_OFF, "almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(tpg_db->al_mac));
				debug(DEBUG_OFF, "itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(tpg_db->itf_mac));
				debug(DEBUG_OFF, "receive_itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(tpg_db->receive_itf_mac));
                if(delete_p1905_neighbor_dev_info(ctx, tpg_db->al_mac,
					(unsigned char *)tpg_db->receive_itf_mac))
                    result = 1;
                /*delete the topology discovery database*/
                LIST_REMOVE(tpg_db, tpddb_entry);
				if (find_discovery_by_almac(ctx, tpg_db->al_mac) == NULL) {
					debug(DEBUG_OFF, "no discovery in data base; del rsp"
						"(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(tpg_db->al_mac));
					delete_exist_topology_response_database(ctx, tpg_db->al_mac);
					del_rsp_cnt++;
				}
				unmask_control_conn_port(ctx, tpg_db->eth_port);
                free(tpg_db);
            }
            tpg_db = tpg_db_temp;
        }
    }

	if (del_rsp_cnt)
		mark_valid_topo_rsp_node(ctx);

    return result;
}

void delete_not_exist_p1905_topology_device(struct p1905_managerd_ctx *ctx, int sec)
{
	struct topology_device_db *tpdev_db, *tpdev_db_temp;

	if (!SLIST_EMPTY(&ctx->topodev_entry.tpdevdb_head)) {
		tpdev_db = SLIST_FIRST(&ctx->topodev_entry.tpdevdb_head);
		while(tpdev_db) {
			tpdev_db_temp = SLIST_NEXT(tpdev_db, tpdev_entry);
			tpdev_db->time_to_live -= sec;
			if(tpdev_db->time_to_live <= 0) {
				debug(DEBUG_TRACE, "dev_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(tpdev_db->aldev_mac));
				SLIST_REMOVE(&ctx->topodev_entry.tpdevdb_head,
					tpdev_db, topology_device_db, tpdev_entry);
				free(tpdev_db);
			}
			tpdev_db = tpdev_db_temp;
		}
	}
}

/**
 *  find remote device's al mac address
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  mac_addr  input, use the sa mac value we got from other unicast message .
 * \param  al_mac_addr  output, provide the al mac
 * \return error code
 */
int find_al_mac_address(struct p1905_managerd_ctx *ctx,
                    unsigned char *mac_addr, unsigned char *al_mac_addr)
{
    struct topology_discovery_db *tpg_db;
    struct topology_response_db *tpgr_db;
    struct device_info_db *dev_db;
    int result = -1;

    /*search in topology discovery database*/
    if(!LIST_EMPTY(&(ctx->topology_entry.tpddb_head)))
    {
        LIST_FOREACH(tpg_db, &(ctx->topology_entry.tpddb_head), tpddb_entry)
        {
            if(!memcmp(mac_addr,tpg_db->al_mac,ETH_ALEN))
            {
                memcpy(al_mac_addr,mac_addr,ETH_ALEN);
                result = 0;
                goto find_finish;
            }
            if(!memcmp(mac_addr,tpg_db->itf_mac,ETH_ALEN))
            {
                memcpy(al_mac_addr,tpg_db->al_mac,ETH_ALEN);
                result = 0;
                goto find_finish;
            }
        }
    }
    /* search in topology reponse database
     * if no matched al mac in topology discovery databse, it means this
     * device is not a neighbor. So we check the topology response database
     * to find any matched deivce
     */
    if(!SLIST_EMPTY(&(ctx->topology_entry.tprdb_head)))
    {
        SLIST_FOREACH(tpgr_db, &(ctx->topology_entry.tprdb_head), tprdb_entry)
        {
            if(!memcmp(mac_addr, tpgr_db->al_mac_addr, ETH_ALEN))
            {
                memcpy(al_mac_addr, mac_addr, ETH_ALEN);
                result = 0;
                goto find_finish;
            }
            if(!SLIST_EMPTY(&(tpgr_db->devinfo_head)))
            {
                SLIST_FOREACH(dev_db, &(tpgr_db->devinfo_head), devinfo_entry)
                {
                    if(!memcmp(mac_addr, dev_db->mac_addr, ETH_ALEN))
                    {
                        memcpy(al_mac_addr, tpgr_db->al_mac_addr, ETH_ALEN);
                        result = 0;
                        goto find_finish;
                    }
                }
            }
        }
    }

find_finish:
    return result;
}

/**
 *  get cmdu message version from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu message version
 */
unsigned char get_mversion_from_msg_hdr(unsigned char *buf)
{
    return (*buf);
}

/**
 *  get cmdu message type from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu message type
 */
unsigned short get_mtype_from_msg_hdr(unsigned char *buf)
{
    unsigned short mtype = 0;

	mtype = *(unsigned short *)(buf+2);
	mtype = be2cpu16(mtype);

    return mtype;
}

/**
 *  get cmdu message type string from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu message type string
 */
char *get_mtype_str_from_msg_hdr(unsigned char *buf)
{
	static char str[64];
	unsigned short mtype = get_mtype_from_msg_hdr(buf);
	snprintf((char *)str,sizeof(str),"Unknow:%04x",mtype);
	switch(mtype)
	{
		case TOPOLOGY_DISCOVERY:
			return "Topology Discovery";
		case TOPOLOGY_NOTIFICATION:
			return "Topology Notification";
		case TOPOLOGY_QUERY:
			return "Topology Query";
		case TOPOLOGY_RESPONSE:
			return "Topology Response";
		case VENDOR_SPECIFIC:
			return "Vendor Specific";
		case LINK_METRICS_QUERY:
			return "Link Metric Query";
		case LINK_METRICS_RESPONSE:
			return "Link Metric Responce";
		case AP_AUTOCONFIG_SEARCH:
			return "AP-autoconfiguration Search";
		case AP_AUTOCONFIG_RESPONSE:
			return "AP-autoconfiguration Responce";
		case AP_AUTOCONFIG_WSC:
			return "AP-autoconfiguration WSC";
		case AP_AUTOCONFIG_RENEW:
			return "AP-autoconfiguration Renew";
		case P1905_PB_EVENT_NOTIFY:
			return "1905.1 Push Button Event";
		case P1905_PB_JOIN_NOTIFY:
			return "1905.1 Push Button Join Notification";
		default:
			return str;//"Unknow";
	}
}

/**
 *  get cmdu message id from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu message id
 */
unsigned short get_mid_from_msg_hdr(unsigned char *buf)
{
    unsigned short mid = 0;

	mid = *(unsigned short *)(buf+4);
	mid = be2cpu16(mid);

    return mid;
}

/**
 *  get cmdu fragment id from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu fragment id
 */
unsigned char get_fid_from_msg_hdr(unsigned char *buf)
{
    return *(buf+6);
}

/**
 *  get cmdu last fragment ind from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu last fragment ind
 */
unsigned char get_last_fragment_ind_from_msg_hdr(unsigned char *buf)
{
    if(((*(buf+7)) & 0x80) == 0x80)
        return 1;
    else
        return 0;
}

/**
 *  get cmdu relay ind from header.
 *
 * \param  buf  input, cmdu message header start address
 * \retun  cmdu relay ind
 */
unsigned char get_relay_ind_from_msg_hdr(unsigned char *buf)
{
    if(((*(buf+7)) & 0x40) == 0x40)
        return 1;
    else
        return 0;
}

/**
 *  check whether receive a relay multicast message
 *
 *
 * \param  relay_ind  input, get from relay ind field of message header
 * \param  mtype  input, get from message type field og message header
 * \return 0: error  1: relay message
 */
int check_relay_message(unsigned char relay_ind,unsigned short mtype)
{
    if(relay_ind)
    {
        if((mtype != TOPOLOGY_NOTIFICATION) && (mtype != VENDOR_SPECIFIC) &&\
           (mtype != AP_AUTOCONFIG_SEARCH) && (mtype != AP_AUTOCONFIG_RENEW) &&\
           (mtype != P1905_PB_EVENT_NOTIFY) && (mtype != P1905_PB_JOIN_NOTIFY))
        {
            return 0;
        }
    }
    return 1;
}

/**
 *  parse topology discovery message
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \param  query  flag for sending topology query
 * \param  notify  flag for sending topology notification
 * \param  al_mac  output al mac address from this message
 * \return error code
 */
int parse_topology_discovery_message(struct p1905_managerd_ctx *ctx,
    unsigned char *buf, unsigned char *query, unsigned char *notify,
    unsigned char *discovery, unsigned char *al_mac, unsigned char *mac_addr,
    unsigned char *smac)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char al_mac_addr[ETH_ALEN];
	unsigned char itf_mac_addr[ETH_ALEN] = {0};
	unsigned char zero_mac[ETH_ALEN] = {0};
	struct topology_discovery_db *tpg_db = NULL;
	unsigned char new_db = 1, ifid = 0, if_sta = 0, if_eth = 0;
	unsigned int integrity = 0;
	unsigned int right_integrity = 0x3;
	int eth_port = -1;
	struct vs_value vs_info;

	os_memset(&vs_info, 0, sizeof(struct vs_value));
	*discovery = 0;
	*query = 0;
	*notify = 0;
	temp_buf = buf;
	while (1) {
	    if( *temp_buf == AL_MAC_ADDR_TLV_TYPE) {
	        integrity|= (1<<0);
	        length = parse_al_mac_addr_type_tlv(temp_buf,al_mac_addr);

	        if (length < 0) {
	            debug(DEBUG_ERROR, "error al mac tlv \n");
	            return -1;
	        }
	        temp_buf += length;
	    } else if (*temp_buf == MAC_ADDR_TLV_TYPE) {
	        integrity|= (1<<1);
	        length = parse_mac_addr_type_tlv(temp_buf,mac_addr);

	        if (length < 0) {
	            debug(DEBUG_ERROR, "error mac tlv \n");
	            return -1;
	        }
	        temp_buf += length;
	    }  else if (*temp_buf == VENDOR_SPECIFIC_TLV_TYPE) {
	        length = parse_vs_tlv(temp_buf, &vs_info);
	        if (length < 0) {
	            debug(DEBUG_ERROR, "error vs tlv \n");
	            return -1;
	        }
	        temp_buf += length;
	    } else if(*temp_buf == END_OF_TLV_TYPE) {
	        break;
	    } else {
	        /*ignore extra tlv*/
	        length = get_cmdu_tlv_length(temp_buf);
	        temp_buf += length;
	    }
	}
	/*check integrity*/
	if (integrity != right_integrity) {
	    debug(DEBUG_ERROR, "incomplete topology discovery 0x%x 0x%x\n",
	            integrity, right_integrity);
	    return -1;
	}


	*discovery = vs_info.discovery;
	os_memcpy(al_mac, al_mac_addr, ETH_ALEN);
	os_memcpy(itf_mac_addr, vs_info.itf_mac, ETH_ALEN);
	if (!os_memcmp(zero_mac, itf_mac_addr, ETH_ALEN)) {
		os_memcpy(itf_mac_addr, ctx->itf[ctx->recent_cmdu_rx_if_number].mac_addr, ETH_ALEN);
	    debug(DEBUG_ERROR, "reset itf_mac_addr(%02x:%02x:%02x:%02x:%02x:%02x)\n",
	            PRINT_MAC(itf_mac_addr));
	} else if (os_memcmp(vs_info.itf_mac, ctx->itf[ctx->recent_cmdu_rx_if_number].mac_addr, ETH_ALEN)) {
		for (ifid = 0; ifid < ctx->itf_number; ifid++) {
			if (!os_memcmp(ctx->itf[ifid].mac_addr, vs_info.itf_mac, ETH_ALEN)) {
				ctx->recent_cmdu_rx_if_number = ifid;
				break;
			}
		}
	}

	for (ifid = 0; ifid < ctx->itf_number; ifid++) {
		if (!os_memcmp(itf_mac_addr, ctx->itf[ifid].mac_addr, ETH_ALEN)) {
			if (ctx->itf[ifid].is_wifi_sta) {
				if (!(ctx->itf[ifid].trx_config & RX_MUL)) {
					debug(DEBUG_ERROR, "%s link down, drop discovery\n",
						ctx->itf[ifid].if_name);
					return -1;
				}
				if_sta = 1;
				break;
			} else if (ctx->itf[ifid].media_type == IEEE802_3_GROUP) {
				if_eth = 1;
			}

			if(ctx->itf[ifid].is_veth == 1) {
				debug(DEBUG_ERROR, "is veth\n");
				if_eth = 0;
			}
		}
	}
	/*for eth device, we should get which switch port this topology discoery comes from*/
	if (if_eth) {
		eth_port = eth_layer_get_client_port_by_mac(al_mac_addr);
		if (eth_port == ETH_ERROR_FAIL) {
			debug(DEBUG_ERROR, "eth_port should not be -1, drop topology discovery\n");
			return -1;
		}
	} else {
		eth_port = -1;
	}

	if (eth_port >= 0 && ctx->conn_port.is_set &&
		ctx->conn_port.port_num == eth_port &&
		os_memcmp(ctx->conn_port.filter_almac, al_mac_addr, ETH_ALEN)) {
		debug(DEBUG_OFF, RED("drop discovery from almac(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
			PRINT_MAC(al_mac_addr));
			return -1;
	}

	if (if_sta && ctx->itf[ifid].vs_info_length >= ETH_ALEN
		&& os_memcmp(mac_addr, ctx->itf[ifid].vs_info, ETH_ALEN) != 0
		&& os_memcmp(ctx->itf[ifid].vs_info, zero_mac, ETH_ALEN) != 0) {
		debug(DEBUG_ERROR, "\tmac addr %02x:%02x:%02x:%02x:%02x:%02x"
			"\tconnected bssid %02x:%02x:%02x:%02x:%02x:%02x not match, drop topology discovery\n",
			PRINT_MAC(mac_addr), PRINT_MAC(ctx->itf[ifid].vs_info));
		*query = 0;
		*notify = 0;
		return 0;
	}

	/*search the AL MAC ADDRESS exist or not*/
	if (!LIST_EMPTY(&ctx->topology_entry.tpddb_head)) {
	    LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
	    {
	        /*compare the AL MAC ADDR and INF MAC ADDR with database*/
	        if (!memcmp(tpg_db->al_mac, al_mac_addr, ETH_ALEN) &&
				!memcmp(tpg_db->itf_mac, mac_addr, ETH_ALEN)) {
	            debug(DEBUG_TRACE, "one al mac and inf mac is matched\n");
	            /*this AL MAC address and INF Mac address exist in database*/
	            new_db = 0;
	            break;
	        }
	    }
	}

	if (new_db) {
#if 0
		/* if sta interface receive a discovery with a different almac compared to the stored one
		 * then need delete the old almac from discovery database, it means this sta interface
		 * connect to a new 1905 device.
		 * if ap interface receive a discovery with a new almac, cannot delete the old almac
		 * from discovery database, because ap interface allow more than one interface connect
		 * to it. need wait timeout to delete old entry from database.
		 */
		if (if_sta) {
			LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
			{
				if (!memcmp(tpg_db->receive_itf_mac, itf_mac_addr, ETH_ALEN)) {
					debug(DEBUG_OFF, GRN("delete recvif_mac(%02x:%02x:%02x:%02x:%02x:%02x) itf_mac(%02x:%02x:%02x:%02x:%02x:%02x) almac(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
						PRINT_MAC(tpg_db->receive_itf_mac), PRINT_MAC(tpg_db->itf_mac), PRINT_MAC(tpg_db->al_mac));
					delete_exist_topology_discovery_database(ctx, tpg_db->al_mac,
						(unsigned char *)tpg_db->itf_mac);
					break;
				}
			}
		}
#endif
	    *query = 1;
		*notify = 1;
	    debug(DEBUG_OFF, "need add new topology_discovery_db database\n");
	    /*no this AL MAC ADDR , add a new topology_discovery_db component in database*/
	    tpg_db = (struct topology_discovery_db *)malloc(sizeof(struct topology_discovery_db));
		if (!tpg_db) {
			debug(DEBUG_ERROR, "alloc tpg_db fail\n");
			return -1;
		}
		memset(tpg_db, 0, sizeof(*tpg_db));
	    memcpy(tpg_db->al_mac, al_mac_addr, ETH_ALEN);
	    memcpy(tpg_db->itf_mac, mac_addr, ETH_ALEN);
	    memcpy(tpg_db->receive_itf_mac, itf_mac_addr, ETH_ALEN);
		tpg_db->time_to_live = TOPOLOGY_DISCOVERY_TTL;
		debug(DEBUG_OFF, "tpg_db->al_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(al_mac_addr));
		debug(DEBUG_OFF, "tpg_db->itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(mac_addr));
		debug(DEBUG_OFF, RED("tpg_db->receive_itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
			PRINT_MAC(itf_mac_addr));
	    LIST_INSERT_HEAD(&ctx->topology_entry.tpddb_head, tpg_db, tpddb_entry);
	} else {
		tpg_db->time_to_live = TOPOLOGY_DISCOVERY_TTL;
	}

	if (tpg_db->eth_port != eth_port)
		/*need update topology response port*/
		*query = 1;
	tpg_db->eth_port = eth_port;

	if (0 > update_p1905_neighbor_dev_info(ctx, al_mac_addr, mac_addr, itf_mac_addr, notify))
	    return -1;

	if (if_eth && eth_port >= 0 && new_db) {
		*notify = 2;
		debug(DEBUG_OFF, RED("new eth conncetion!\n"));
	}

	if (*notify)
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
		   	ctx->service, &ctx->ap_cap_entry.oper_bss_head,
		    &ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);

	debug(DEBUG_TRACE, "receive discovery from (%02x:%02x:%02x:%02x:%02x:%02x) "
		"receive itf(%02x:%02x:%02x:%02x:%02x:%02x),  eth_port=%d vs_disc(%d)\n",
		PRINT_MAC(tpg_db->al_mac), PRINT_MAC(tpg_db->receive_itf_mac),
		tpg_db->eth_port, *discovery);

	return 0;
}

/**
 *  parse topology notification message
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \param  al_mac  output al mac address from this message
 * \return error code
 */
int parse_topology_notification_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf, unsigned char *al_mac, unsigned char *bssid,
	unsigned char *mac, unsigned char *event, unsigned char *neth_almac)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char al_mac_addr[ETH_ALEN];
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x1;
	struct vs_value vs_info;

	os_memset(&vs_info, 0, sizeof(struct vs_value));
    temp_buf = buf;
	reset_stored_tlv(ctx);
    while (1) {
        if (*temp_buf == AL_MAC_ADDR_TLV_TYPE) {
            integrity |= (1<<0);
            length=parse_al_mac_addr_type_tlv(temp_buf, al_mac_addr);

            if (length < 0) {
                debug(DEBUG_ERROR, "error al mac tlv \n");
                return -1;
            }
			if (!memcmp(ctx->p1905_al_mac_addr, al_mac_addr, ETH_ALEN))
				return -1;
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        } else if (*temp_buf == CLIENT_ASSOCIATION_EVENT_TYPE) {
			length = parse_client_assoc_event_tlv(temp_buf, mac, bssid, event);

            if(length < 0) {
                debug(DEBUG_ERROR, "error client_assoc_event tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}

            temp_buf += length;
		} else if (*temp_buf == VENDOR_SPECIFIC_TLV_TYPE) {
	        length = parse_vs_tlv(temp_buf, &vs_info);

	        if (length < 0) {
	            debug(DEBUG_ERROR, "error vs tlv \n");
	            return -1;
	        }
	        temp_buf += length;
	    } else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
		} else {
            debug(DEBUG_ERROR, "wrong TLV in topology notification message\n");
            /*ignore extra tlv*/
            length = get_cmdu_tlv_length(temp_buf);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
    }

    /*check integrity*/
    if (integrity != right_integrity) {
        debug(DEBUG_ERROR, "incomplete topology notification 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }

    os_memcpy(al_mac, al_mac_addr, ETH_ALEN);
	os_memcpy(neth_almac, vs_info.neth_almac, ETH_ALEN);
    return 0;
}

#ifdef MAP_R2
int parse_map_version_tlv(unsigned char *buf,
	unsigned char *almac)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == MULTI_AP_VERSION_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;

	return (length+3);
}

int parse_topology_query_message(struct p1905_managerd_ctx *ctx,
                                unsigned char *buf, unsigned char *almac)

{
	int length =0;
	unsigned char *temp_buf;

	temp_buf = buf;

	while(1) {
		if (*temp_buf == MULTI_AP_VERSION_TYPE) {
			length = parse_map_version_tlv(temp_buf, almac);
			if(length < 0) {
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}
			temp_buf += length;
			break;
		}else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	return 0;
}
#endif

/**
 *  parse topology notification message
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \return error code
 */
int parse_topology_response_message(struct p1905_managerd_ctx *ctx,
                                unsigned char *buf, unsigned char *almac)
{
    int length =0;
    unsigned char *temp_buf;
    unsigned char al_mac_addr[ETH_ALEN];
    struct topology_response_db *tpgr_db;
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x3;

    temp_buf = buf;

    /*we must get the AL MAC addr to get correct database pointer
    , so need to search device info tlv  firstly*/
    while (1) {
        if ((*temp_buf) == DEVICE_INFO_TLV_TYPE) {
            integrity |= (1 << 0);
        } else if (*temp_buf == SUPPORTED_SERVICE_TLV_TYPE) {
			integrity |= (1 << 1);
		} else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
		}
		//need to implement error handling
		length = get_cmdu_tlv_length(temp_buf);
		temp_buf += length;
    }

    /*check integrity*/
    if (integrity != right_integrity) {
        debug(DEBUG_ERROR, "incomplete topology response 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }

	temp_buf = buf;

    temp_buf +=3;//shift to AL MAC addr field in device information tlv
    memcpy(al_mac_addr,temp_buf,ETH_ALEN);
    memcpy(almac,al_mac_addr,ETH_ALEN);

	debug(DEBUG_TRACE, "recv TOPOLOGY_response from %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac_addr));

    /*delete the database if already exist*/
    tpgr_db = delete_exist_topology_response_content(ctx, al_mac_addr);
	if (!tpgr_db) {
		/*add a new topology_response_db component in database*/
	    tpgr_db = (struct topology_response_db *)malloc(sizeof(struct topology_response_db));
		if (tpgr_db == NULL) {
			debug(DEBUG_ERROR, "allocate memory fail\n");
			return -1;
		}
		os_memcpy(tpgr_db->al_mac_addr, al_mac_addr, ETH_ALEN);
		SLIST_INSERT_HEAD(&(ctx->topology_entry.tprdb_head), tpgr_db, tprdb_entry);
	}
	os_get_time(&tpgr_db->last_seen);
	tpgr_db->recv_ifid = ctx->recent_cmdu_rx_if_number;
	tpgr_db->valid = 1;
    /*init its own device information list and bridge capacity list*/
    SLIST_INIT(&tpgr_db->devinfo_head);
    LIST_INIT(&tpgr_db->brcap_head);

    /* until now, we get the correct pointer
     * Move to the start of buf, then do tlvs parsing
     */
    temp_buf = buf;
	reset_stored_tlv(ctx);
    while(1)
    {
        if(*temp_buf == DEVICE_INFO_TLV_TYPE)
        {
        	/*then shift back to start of device info tlv, parsing it first*/
		    length = parse_device_info_type_tlv(ctx, temp_buf, &(tpgr_db->devinfo_head));
		    if(length < 0)
		    {
		        debug(DEBUG_ERROR, "error device info tlv\n");
		        return -1;
		    }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            /*we have parsed this tlv, so just get length & shift*/
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
        else if(*temp_buf == BRIDGE_CAPABILITY_TLV_TYPE)
        {
            length=parse_bridge_capability_type_tlv(temp_buf,
                   &(tpgr_db->brcap_head));
            if(length < 0)
            {
                debug(DEBUG_ERROR, "error bridge capability tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
        else if(*temp_buf == P1905_NEIGHBOR_DEV_TLV_TYPE)
        {
            length=parse_p1905_neighbor_device_type_tlv(temp_buf,
                    &(tpgr_db->devinfo_head));
            if(length < 0)
            {
                debug(DEBUG_ERROR, "error p1905.1 neighbor device tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
        else if(*temp_buf == NON_P1905_NEIGHBOR_DEV_TLV_TYPE)
        {
            length=parse_non_p1905_neighbor_device_type_tlv(temp_buf,
                    &(tpgr_db->devinfo_head));
            if(length < 0)
            {
                debug(DEBUG_ERROR, "error non-p1905.1 neighbor device tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
		else if (*temp_buf == SUPPORTED_SERVICE_TLV_TYPE) {
            length = parse_supported_service_tlv(temp_buf, &tpgr_db->support_service);
            if (length < 0) {
                debug(DEBUG_ERROR, "error ap_operational_bss tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
#ifdef MAP_R2
		else if (*temp_buf == MULTI_AP_VERSION_TYPE) {
			length = parse_map_version_tlv(temp_buf, almac);
			if(length < 0) {
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}
			temp_buf += length;
		}
#endif
        else if(*temp_buf == END_OF_TLV_TYPE)
            break;
        else
        {
            /*ignore extra tlv*/
            length = get_cmdu_tlv_length(temp_buf);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
    }

	if (ctx->itf[ctx->recent_cmdu_rx_if_number].media_type == IEEE802_3_GROUP) {
		tpgr_db->eth_port = eth_layer_get_client_port_by_mac(al_mac_addr);
		if (tpgr_db->eth_port == ETH_ERROR_FAIL)
			debug(DEBUG_ERROR, "BUG!tpgr_db->eth_port should not be -1\n");
	} else
		tpgr_db->eth_port = -1;

	debug(DEBUG_WARN, RED("tpgr_db eth_port=%d\n"),tpgr_db->eth_port);
	/*shall not enable PON with MAP feature in Certification case*/
	if (!ctx->MAP_Cer)
		mask_control_conn_port(ctx, tpgr_db);

    return 0;
}

void query_neighbor_info(struct p1905_managerd_ctx *ctx, unsigned char *almac,
		unsigned char if_number)
{
	struct topology_response_db *tpgr = NULL, *tpgr_tmp = NULL;
    struct device_info_db *dev_info = NULL;
	struct p1905_neighbor_device_db *dev = NULL;
	unsigned char exist = 0;

	SLIST_FOREACH(tpgr, &ctx->topology_entry.tprdb_head, tprdb_entry)
	{
		if(!os_memcmp(tpgr->al_mac_addr, almac, 6))
			break;
	}

	if (!tpgr) {
		debug(DEBUG_ERROR, "something wrong! almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(almac));
		return;
	}

    SLIST_FOREACH(dev_info, &tpgr->devinfo_head, devinfo_entry)
    {
		SLIST_FOREACH(dev, &(dev_info->p1905_nbrdb_head), p1905_nbrdb_entry)
		{
			if (!os_memcmp(dev->p1905_neighbor_al_mac, ctx->p1905_al_mac_addr, 6)) {
				debug(DEBUG_INFO, "not send query to self(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(dev->p1905_neighbor_al_mac));
				continue;
			}

			exist = 0;
			SLIST_FOREACH(tpgr_tmp, &ctx->topology_entry.tprdb_head, tprdb_entry)
			{
				if (os_memcmp(tpgr_tmp->al_mac_addr, almac, 6)) {
					if(!os_memcmp(tpgr_tmp->al_mac_addr, dev->p1905_neighbor_al_mac, 6) &&
						(tpgr_tmp->valid == 1)) {
						debug(DEBUG_INFO, "topo exist almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
							PRINT_MAC(dev->p1905_neighbor_al_mac));
						exist = 1;
						break;
					}
				}
			}
			if (!exist) {
				insert_cmdu_txq(dev->p1905_neighbor_al_mac, ctx->p1905_al_mac_addr,
                	e_topology_query, ++ctx->mid, ctx->itf[if_number].if_name, 0);
				debug(DEBUG_OFF, "send topology query to almac(%02x:%02x:%02x:%02x:%02x:%02x)"
					" mid(%04x)\n",
					PRINT_MAC(dev->p1905_neighbor_al_mac), ctx->mid);
			}
		}
    }
}
unsigned char is_internal_used_vendor_specific(unsigned char *vs_info)
{
	unsigned char charator_value[4] = {0xFF, 0x00, 0x0C, 0xE7};
	/*0xff, 0x00, 0x0C, 0xE7*/

	if ( !os_memcmp(vs_info, charator_value, sizeof(charator_value)))
		return 1;
	else
		return 0;
}
/**
 *  parse vendor specific message. It for PC tool use.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  vs_info  input
 * \return error code
 */
int parse_vendor_specific_message(struct p1905_managerd_ctx* ctx, unsigned char *buf,
                                    struct p1905_vs_info *vs_info, unsigned char *internal_vendor_message)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x1;

    temp_buf = buf;
	reset_stored_tlv(ctx);
	*internal_vendor_message = 0;
    while (1) {
        if (*temp_buf == VENDOR_SPECIFIC_TLV_TYPE) {
            integrity |= (1 << 0);
            length=parse_vendor_specific_type_tlv(temp_buf,vs_info);

            if (length < 0) {
                debug(DEBUG_ERROR, "error vs tlv\n");
                return -1;
            }
			if (length >= 8 && is_internal_used_vendor_specific(temp_buf + 6)) {
				*internal_vendor_message = 1;
				debug(DEBUG_OFF, "internal used vendor specific message\n");
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
            temp_buf += length;
        }
        else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
        } else {
            /* ignore extra tlv, although vendor specific cmdu can include any
             * type tlvs, but we have no definition about it.
             */
            length = get_cmdu_tlv_length(temp_buf);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
    }

    /*check integrity*/
    if (integrity != right_integrity) {
        debug(DEBUG_ERROR, "incomplete vendor specific 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }


    return 0;
}

/**
 *  parse 1905 internal using vendor specific message. It for PC tool use.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  vs_info  input
 * \return error code
 */
int parse_1905_internal_vendor_specific_message(struct p1905_managerd_ctx* ctx, unsigned char *buf)
{
	unsigned char *temp_buf;
	unsigned char need_query = 0, need_notify = 0, discvoery = 0;
	unsigned char peer_al_mac[ETH_ALEN] = {0x00};
	unsigned char peer_itf_mac[ETH_ALEN] = {0x00};
	int j = 0;

	/*1905 internal using vendor specific tlv format
	   type			1 byte			0x11
	   length			2 bytes			0x, 0x
	   oui			3 bytes			0x00, 0x0C, 0xE7
	   function type	1 byte			0xff
	   suboui			3 bytes			0x00, 0x0C, 0xE7
	   sub func type	1 byte			0x
	*/
    temp_buf = buf + 10;

	debug(DEBUG_ERROR, "vendor specific internal message, subtype=%02x\n", *temp_buf);
	switch (*temp_buf) {
		case internal_vendor_discovery_message_subtype:
			{
				if (0 > parse_topology_discovery_message(ctx, buf,
	        		&need_query, &need_notify, &discvoery, peer_al_mac,
	        		peer_itf_mac, NULL)) {
		            debug(DEBUG_ERROR, "receive error vendor specific topology discovery message\n");
		            break;
		        }

				for (j = 1; j < ctx->itf_number; j++) {
				/*send multicast cmdu to all the ethenet interface */
					if (ctx->itf[j].media_type == IEEE802_3_GROUP) {
						debug(DEBUG_OFF, "receive specific discovery, reply topology discovery by interface[%s]\n",
							ctx->itf[j].if_name);
						insert_cmdu_txq((unsigned char*)p1905_multicast_address, ctx->p1905_al_mac_addr,
							e_topology_discovery, ++ctx->mid, ctx->itf[j].if_name, 0);
					}
				}
				topology_discovery_action(ctx, peer_al_mac, need_query, need_notify);
				update_topology_tree(ctx);
			}
			break;
		default:
			debug(DEBUG_ERROR, "un-supported internal vendor message subtype\n");
			break;
	}

    return 0;
}


/**
 *  parse link metric query message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \return error code
 */
int parse_link_metric_query_message(struct p1905_managerd_ctx *ctx,
                                        unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x1;

    temp_buf = buf;

	reset_stored_tlv(ctx);

    while (1) {
        if (*temp_buf == LINK_METRICS_QUERY_TLV_TYPE) {
            integrity |= (1 << 0);

            length=parse_link_metric_query_type_tlv(temp_buf,
                    ctx->link_metric_response_para.target,
                    &(ctx->link_metric_response_para.type));

            if(length < 0) {
                debug(DEBUG_ERROR, "error link metric query tlv \n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
        else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
        }  else {
            debug(DEBUG_ERROR, "ignore TLV in link metric query message\n");
            /*ignore extra tlv*/
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }

    /*check integrity*/
    if (integrity != right_integrity) {
        debug(DEBUG_ERROR, "incomplete link metrics query 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }

    return 0;
}

#ifdef SUPPORT_ALME
/**
 *  parse link metric response message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \param  metric_rsp  pointer of alme response buffer
 * \return error code
 */

int parse_link_metric_response_message(struct p1905_managerd_ctx *ctx,
            unsigned char *buf, ALME_GET_METRIC_RSP *metric_rsp)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned char got_tx_link_tlv = 0, got_rx_link_tlv = 0;

    temp_buf = buf;

    /* in implementation, I use BOTH_TX_AND_RX_METRICS to query link metrics.
     * so I must get both RX & TX metrics tlv in this message.
     * Based on this concept, I parse tx link metrics tlv first to get the number
     * of connecting interfaces and then parse rx link metrics tlv.
     */
    while(1)
    {
        if(*temp_buf == TRANSMITTER_LINK_METRIC_TYPE)
        {
            length = parse_transmitter_link_metrics_type_tlv(temp_buf,
                ctx->p1905_al_mac_addr, metric_rsp);

            if(length < 0)
            {
                debug(DEBUG_ERROR, "error parse transmitted link metrics tlv type\n");
                return -1;
            }

            got_tx_link_tlv = 1;
            temp_buf += length;
        }
        else if(*temp_buf == RESULT_CODE_TLV_TYPE)
        {
            length = parse_link_metrics_result_code_type_tlv(temp_buf);

            if(length < 0)
            {
                debug(DEBUG_ERROR, "link metrics response with invalid neighbor\n");
                return -1;
            }
            temp_buf += length;
        }
        else if(*temp_buf == END_OF_TLV_TYPE)
            break;
        else
        {
            /*ignore other tlv*/
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }
    if(!got_tx_link_tlv)
    {
        debug(DEBUG_ERROR, "no tx link metrics tlv\n");
        return -1;
    }

    /*return to start position again to parse rx link metrics tlv*/
    temp_buf = buf;
    while(1)
    {
        if(*temp_buf == RECEIVER_LINK_METRIC_TYPE)
        {
            length = parse_receiver_link_metrics_type_tlv(temp_buf,
                ctx->p1905_al_mac_addr, metric_rsp);

            if(length < 0)
            {
                debug(DEBUG_ERROR, "error parse receiver link metrics tlv type\n");
                return -1;
            }

            got_rx_link_tlv = 1;
            temp_buf += length;
        }
        else if(*temp_buf == END_OF_TLV_TYPE)
            break;
        else
        {
            /*ignore other tlv*/
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }
    if(!got_rx_link_tlv)
    {
        debug(DEBUG_ERROR, "no rx link metrics tlv\n");
        return -1;
    }

    /* no integrity check because we already used got_tx_link_tlv and
     * got_rx_link_tlv to check
     */

    return 0;
}
#endif

/**
 *  parse auto configuration search message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \param  al_mac output, gotten from the al mac type tlv
 * \return error code
 */
int parse_ap_autoconfig_search_message
    (struct p1905_managerd_ctx *ctx, unsigned char *buf, unsigned char *al_mac)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x7;

    temp_buf = buf;

    while (1) {
        if (*temp_buf == AL_MAC_ADDR_TLV_TYPE) {
            integrity |= (1 << 0);

            length = parse_al_mac_addr_type_tlv(temp_buf, al_mac);

            if (length < 0) {
                debug(DEBUG_ERROR, "error al mac tlv\n");
                return -1;
            }
			if (!memcmp(ctx->p1905_al_mac_addr, al_mac, ETH_ALEN)) {
                debug(DEBUG_ERROR, "receive search sent by myself\n");
				return -1;
			}
            temp_buf += length;
        } else if (*temp_buf == SEARCH_ROLE_TLV_TYPE) {
            integrity |= (1 << 1);

            length = parse_search_role_tlv(temp_buf);
            if (length < 0) {
                debug(DEBUG_ERROR, "error search role tlv\n");
                return -1;
            }
            temp_buf += length;
        }
        else if (*temp_buf == AUTO_CONFIG_FREQ_BAND_TLV_TYPE) {
            integrity |= (1 << 2);

            length = parse_auto_config_freq_band_tlv(temp_buf, &ctx->peer_search_band);
            if (length < 0) {
                debug(DEBUG_ERROR, "error freq tlv\n");
                return -1;
            }
            temp_buf += length;
        }
#ifdef MAP_R2
		 else if (*temp_buf == MULTI_AP_VERSION_TYPE) {
			length = parse_map_version_tlv(temp_buf, al_mac);
			if(length < 0) {
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}
			temp_buf += length;
		}
#endif
        else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
        } else {
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }

    /*check integrity*/
    if(integrity != right_integrity)
    {
        debug(DEBUG_ERROR, "incomplete ap auto config search 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }
    return 0;
}

/**
 *  parse auto configuration response message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \return error code
 */

int parse_ap_autoconfig_response_message
    (struct p1905_managerd_ctx *ctx, unsigned char *buf, unsigned char *band,
    	unsigned char* service, unsigned char* role_register, unsigned char *al_mac)
{
	int length = 0;
	unsigned char *temp_buf = NULL;
    unsigned int integrity = 0;
	unsigned int right_integrity = 0x7;

	temp_buf = buf;
	reset_stored_tlv(ctx);
    while (1) {
        if (*temp_buf == SUPPORT_ROLE_TLV_TYPE) {
            integrity |= (1 << 0);

            length = parse_supported_role_tlv(temp_buf);
            if (length < 0) {
                debug(DEBUG_ERROR, "error support role tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			*role_register = 1;
            temp_buf += length;
        } else if(*temp_buf == SUPPORT_FREQ_BAND_TLV_TYPE) {
            integrity |= (1 << 1);

            length = parse_supported_freq_band_tlv(temp_buf, band);
            if (length < 0) {
                debug(DEBUG_ERROR, "error support freq tlv\n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
            temp_buf += length;
        } else if (*temp_buf == SUPPORTED_SERVICE_TLV_TYPE) {
            integrity |= (1 << 2);

            length = parse_supported_service_tlv(temp_buf, service);
            if (length < 0) {
                debug(DEBUG_ERROR, "error support service tlv \n");
                return -1;
            }
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
        }
#ifdef MAP_R2
		else if (*temp_buf == MULTI_AP_VERSION_TYPE) {
			length = parse_map_version_tlv(temp_buf, al_mac);
			if(length < 0) {
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}
			temp_buf += length;
		}
#endif
		else if(*temp_buf == END_OF_TLV_TYPE) {
            break;
        } else {
            length = get_cmdu_tlv_length(temp_buf);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, (unsigned short)length);
			}
            temp_buf += length;
        }
    }

    /*check integrity*/
    if(integrity != right_integrity)
    {
        debug(DEBUG_ERROR, "incomplete ap auto config response 0x%x 0x%x\n",
                integrity, right_integrity);
        return -1;
    }
    return 0;
}

int parse_ap_autoconfig_renew_message
    (struct p1905_managerd_ctx *ctx, unsigned char *buf, unsigned char *al_mac, unsigned char* band)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned int integrity = 0;
    unsigned int right_integrity = 0x7;

    temp_buf = buf;

    while (1) {
        if (*temp_buf == AL_MAC_ADDR_TLV_TYPE) {
            integrity |= (1 << 0);

            length = parse_al_mac_addr_type_tlv(temp_buf, al_mac);

            if (length < 0) {
                debug(DEBUG_ERROR, "error al mac tlv\n");
                return -1;
            }
			if (!memcmp(ctx->p1905_al_mac_addr, al_mac, ETH_ALEN))
				return -1;
            temp_buf += length;
        } else if (*temp_buf == SUPPORT_ROLE_TLV_TYPE) {
            integrity |= (1 << 1);

            length = parse_supported_role_tlv(temp_buf);
            if(length < 0) {
                debug(DEBUG_ERROR, "error support role tlv\n");
                return -1;
            }
            temp_buf += length;
        }
        else if (*temp_buf == SUPPORT_FREQ_BAND_TLV_TYPE) {
            integrity |= (1 << 2);

            length = parse_supported_freq_band_tlv(temp_buf, band);
            if (length < 0) {
                debug(DEBUG_ERROR, "error support freq tlv\n");
                return -1;
            }
            temp_buf += length;
        }
        else if (*temp_buf == END_OF_TLV_TYPE) {
            break;
        } else {
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }

    /*check integrity*/
    if (integrity != right_integrity) {
        debug(DEBUG_ERROR, "incomplete ap auto config renew 0x%x 0x%x\n",
                integrity, right_integrity );
        return -1;
    }

    return 0;
}

/**
 *  parse wsc M1/M2 message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, tlvs start position of receive cmdu packet
 * \return error code
 */

int parse_ap_autoconfig_wsc_message(struct p1905_managerd_ctx *ctx,
	unsigned char *almac, unsigned char *buf, unsigned char *radio)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int rc_integrity = 0x1F;
	unsigned int rm_integrity = 0x1F;
	unsigned int right_integrity = 0;
	int m2_num = 0;
	struct agent_list_db *agent_info = NULL;

    temp_buf = buf;
	ctx->peer_bss_need_config = 0;
	ctx->current_bss_config_index = 0;
	ctx->bss_success_config = 0;

	right_integrity = (ctx->role == CONTROLLER) ? rm_integrity : rc_integrity;

    while (1) {
        if (*temp_buf == WSC_TLV_TYPE) {
            integrity |= (1 << 0);
			if (ctx->role == AGENT)
				memset(&ctx->ap_config_data, 0, sizeof(WSC_CONFIG));
            length = parse_wsc_tlv(temp_buf, ctx);
            if (length < 0) {
                debug(DEBUG_ERROR, "error wsc tlv\n");
                return -1;
            }
			/*copy all setting from each M2*/
			if (ctx->role == AGENT) {
				memcpy(&(ctx->current_autoconfig_info.config_data[m2_num]),
					&ctx->ap_config_data, sizeof(WSC_CONFIG));
				m2_num++;
			}
            temp_buf += length;
        } else if (*temp_buf == AP_RADIO_IDENTIFIER_TYPE) {
			integrity |= (1 << 1);
			length = parse_ap_radio_identifier_tlv(temp_buf, ctx
#ifdef MAP_R2
			, 0
#endif // MAP_R2
			);
			if(length < 0) {
				debug(DEBUG_ERROR, "error ap radio identifier tlv\n");
				return -1;
			}
			temp_buf += length;
		} else if(*temp_buf == AP_RADIO_BASIC_CAPABILITY_TYPE) {
			integrity |= (1 << 2);
			length = parse_ap_radio_basic_cap_tlv(ctx, temp_buf, ctx->identifier,
				&ctx->peer_bss_need_config, &ctx->band_cap);
			if(length < 0) {
				debug(DEBUG_ERROR, "error ap radio identifier tlv\n");
				return -1;
			}
			/*fill in agent radio capability*/
			find_agent_info(ctx, almac, &agent_info);
			if (!agent_info) {
				debug(DEBUG_ERROR, "error! no agent info exist insert it\n");
				insert_agent_info(ctx, almac);
				find_agent_info(ctx, almac, &agent_info);
			}
			switch (ctx->band_cap) {
			case BAND_2G_CAP:
				agent_info->radio_cap |= RADIO_2G_CAP;
				*radio = RADIO_2G_CAP;
			break;
			case BAND_5GL_CAP:
				agent_info->radio_cap |= RADIO_5GL_CAP;
				*radio = RADIO_5GL_CAP;
			break;
			case BAND_5GH_CAP:
				agent_info->radio_cap |= RADIO_5GH_CAP;
				*radio = RADIO_5GH_CAP;
			break;
			case BAND_5G_CAP:
				agent_info->radio_cap |= RADIO_5G_CAP;
				*radio = RADIO_5G_CAP;
			break;
			default:
			break;
			}
			agent_info->band_cap |= ctx->band_cap;
			debug(DEBUG_TRACE, "agent(%02x:%02x:%02x:%02x:%02x:%02x) radio_cap(%02x)"
				" band_cap(%02x) radio(%02x)\n", PRINT_MAC(almac),
				agent_info->radio_cap, agent_info->band_cap, *radio);
			debug(DEBUG_TRACE, "peer ap radio basic cap, max bss number=%d\n",
				ctx->peer_bss_need_config);
			temp_buf += length;
		}
#ifdef MAP_R2
		else if (*temp_buf == DEFAULT_8021Q_SETTING_TYPE) {
			integrity |= (1 << 3);
			ctx->map_policy.setting.updated = 1;
			length = parse_default_802_1q_setting_tlv(temp_buf, ctx);
			debug(DEBUG_ERROR, "%s x length:%d\n", __func__, length);
			temp_buf += length;
			debug(DEBUG_ERROR, "%s xx length:%d\n", __func__, length);
		} else if (*temp_buf == TRAFFIC_SEPARATION_POLICY_TYPE) {
			integrity |= (1 << 4);
			ctx->map_policy.tpolicy.updated = 1;
			/*delete the previously reserved traffic policy*/
			delete_exist_traffic_policy(ctx, &ctx->map_policy.tpolicy);

			length = parse_traffic_separation_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "%s error trffic separation policy tlv\n", __func__);
				return -1;
			}
			temp_buf += length;
		} else if (*temp_buf == R2_AP_CAPABILITY_TYPE) {
			// TODO: store R2 ap cap
			integrity |= (1 << 5);

			/*jump to tlv len field*/
			length = *(unsigned short *)(temp_buf + 1);
			/*endian change*/
			length = be2cpu16(length);
			/*add byte of tlytyle & tlvlength*/
			length += 3;

			//store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
#endif // #ifdef MAP_R2
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
        }
    }

	/*check integrity*/
	if(!(integrity & right_integrity)) {
		debug(DEBUG_ERROR, "incomplete ap auto config WSC 0x%x 0x%x\n",
			integrity,right_integrity);
		return -1;
	}

	if (ctx->role == AGENT) {
		ctx->current_autoconfig_info.config_number = m2_num;
	}

    return 0;
}

int parse_client_capability_query_message(
	struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int right_integrity = 0x1;

	temp_buf = buf;

	while (1) {
		if (*temp_buf == CLIENT_INFO_TYPE) {
			integrity |= (1 << 0);

			length = parse_client_info_tlv(temp_buf, ctx, ctx->sinfo.cinfo.bssid, ctx->sinfo.cinfo.sta_mac);

			if (length < 0) {
				debug(DEBUG_ERROR, "error client info tlv\n");
				return -1;
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	/*check integrity*/
	if (integrity != right_integrity) {
		debug(DEBUG_ERROR, "incomplete client capablilty query 0x%x 0x%x\n",
				integrity, right_integrity);
		return -1;
	}

	return 0;
}

int check_recv_mid(struct p1905_managerd_ctx *ctx,
	unsigned char *salmac, unsigned short mid, unsigned short mtype)
{
	struct topology_device_db *tpdev_db = NULL, *exist_tpdev_db = NULL;
	unsigned short i = 0;

	if((mtype == TOPOLOGY_RESPONSE) || (mtype == LINK_METRICS_RESPONSE) ||
       (mtype == AP_AUTOCONFIG_RESPONSE) || (mtype == AP_CAPABILITY_REPORT) ||
       (mtype == CHANNLE_PREFERENCE_REPORT) || (mtype == CHANNLE_SELECTION_RESPONSE) ||
       (mtype == CLIENT_CAPABILITY_REPORT) || (mtype == AP_LINK_METRICS_RESPONSE) ||
       (mtype == ASSOC_STA_LINK_METRICS_RESPONSE) || (mtype == Ack_1905)) {
            return 0;
	}

	if(!SLIST_EMPTY(&ctx->topodev_entry.tpdevdb_head)) {
		SLIST_FOREACH(tpdev_db, &ctx->topodev_entry.tpdevdb_head, tpdev_entry)
		{
			if(!memcmp(tpdev_db->aldev_mac, salmac, ETH_ALEN)) {
				exist_tpdev_db = tpdev_db;
				break;
			}
		}
	}
	if(!exist_tpdev_db)
	{
		exist_tpdev_db = (struct topology_device_db *)malloc(sizeof(struct topology_device_db));
		if(exist_tpdev_db == NULL)
			return -1;
		memset(exist_tpdev_db, 0, sizeof(struct topology_device_db));
		memcpy(exist_tpdev_db->aldev_mac, salmac, ETH_ALEN);
		exist_tpdev_db->time_to_live = TOPOLOGY_DEVICE_TTL;
		exist_tpdev_db->mid_list[0] = mid;
		exist_tpdev_db->mid_set = 0;
		exist_tpdev_db->recv_mid = mid;
		debug(DEBUG_TRACE, "first time recv mid(%u) aldev_mac(%02x:%02x:%02x:%02x:%02x:%02x) msgtype=0x%02x\n",
			mid, PRINT_MAC(salmac),mtype);
		SLIST_INSERT_HEAD(&ctx->topodev_entry.tpdevdb_head, exist_tpdev_db, tpdev_entry);
	} else {
		for(i = 0; i < MAX_MID_QUEUE; i++) {
			if (mid == exist_tpdev_db->mid_list[i]) {
				debug(DEBUG_ERROR, "drop mid(%04x) aldev_mac(%02x:%02x:%02x:%02x:%02x:%02x) msgtype=0x%04x\n",
					mid, PRINT_MAC(salmac), mtype);
				return -1;
			}
		}
		exist_tpdev_db->time_to_live = TOPOLOGY_DEVICE_TTL;
		exist_tpdev_db->recv_mid = mid;
		exist_tpdev_db->mid_set++;
		if (exist_tpdev_db->mid_set == MAX_MID_QUEUE)
			exist_tpdev_db->mid_set = 0;
		exist_tpdev_db->mid_list[exist_tpdev_db->mid_set] = mid;
	}

	return 0;
}

void relay_fragment_send(struct p1905_managerd_ctx *ctx,
		unsigned char *tlv, unsigned int tlv_len,
		unsigned char *eth_hdr, unsigned char *almac)
{
	unsigned short msg_len = 0;
	unsigned char *msg = NULL;
	int i = 0;

	msg_len = ETH_HLEN + CMDU_HLEN + tlv_len;
	msg = (unsigned char *)malloc(msg_len);
	if (!msg) {
		debug(DEBUG_ERROR, "allco msg fail\n");
		return;
	}

	memcpy(msg, eth_hdr, (ETH_HLEN + CMDU_HLEN));
	memcpy((msg + ETH_ALEN), almac, ETH_ALEN);
	memcpy((msg + ETH_ALEN + CMDU_HLEN), tlv, tlv_len);

	for (i = (ctx->recent_cmdu_rx_if_number == FIRST_VITUAL_ITF || ctx->concurrent == 0) ? 1 : 0 ;
		i < ctx->itf_number; i++) {
        /*do not relay to original receive interface*/
        if (i != ctx->recent_cmdu_rx_if_number) {
			if (tlv_len > MAX_TLVS_LENGTH) {
				if (0 > cmdu_tx_fragment(ctx, msg, msg_len, i)) {
					debug(DEBUG_ERROR, "fragment send failed on %s (%s)",
						ctx->itf[i].if_name, strerror(errno));
					free(msg);
					return;
				}
        	}
    	}
	}

	free(msg);
}


/**
 *  parse cmdu message.
 *
 *
 * \param  ctx  1905.1 managerd context
 * \param  buf  input, cmdu message start address(header + payload)
 * \param  dmac  input, dmac of receive packet
 * \param  smac  input, smac of receive packet
 * \param  len   input, length of received packet
 * \return error code
 */
int parse_cmdu_message(struct p1905_managerd_ctx *ctx, unsigned char *buf,
                       unsigned char *dmac, unsigned char *smac,
                       int len)
{
    unsigned char *temp_buf;
    unsigned char mversion;
    unsigned short mtype;
    unsigned short mid = 0;
    unsigned char fid;
    unsigned char last_fragment_ind;
    unsigned char relay_ind;
    unsigned char al_mac[ETH_ALEN] = {0};
    unsigned char itf_mac[ETH_ALEN] = {0};
    unsigned char need_query = 0, need_notify = 0, need_discovery = 0, internal_vendor_message = 0;
    int tlv_len = 0, temp_tlv_len = 0;
    unsigned short queue_len = 0;
    unsigned char *tlv_start;
    struct p1905_vs_info vs_info;
	unsigned char band = 0, role = 0, service = 0, radio = 0;
//#ifdef SUPPORT_CMDU_RELIABLE
//	struct topology_response_db *tpgr_db = NULL;
//#endif
	unsigned char multicast_address[ETH_ALEN] = {0};

	memcpy(multicast_address, p1905_multicast_address, ETH_ALEN);
    /*get message info from message header*/
    temp_buf = buf;
    tlv_start = buf + 8;
    mversion = get_mversion_from_msg_hdr(temp_buf);
    mtype = get_mtype_from_msg_hdr(temp_buf);
    mid = get_mid_from_msg_hdr(temp_buf);
    fid = get_fid_from_msg_hdr(temp_buf);
    last_fragment_ind = get_last_fragment_ind_from_msg_hdr(temp_buf);
    relay_ind = get_relay_ind_from_msg_hdr(temp_buf);

    debug(DEBUG_TRACE, "cmdu message type = 0x%x\n", mtype);
    debug(DEBUG_TRACE, "cmdu message id = 0x%04x\n", mid);
    debug(DEBUG_TRACE, "cmdu message fid = %d\n", fid);
    debug(DEBUG_TRACE, "cmdu message last fragment ind = %d\n", last_fragment_ind);

    /*deal with message version, it must be 0x00*/
    if(mversion != MESSAGE_VERSION)
        return 0;

	os_memcpy(al_mac, smac, ETH_ALEN);

    /*if relay indicator =1 but this is not relay multicast message, ignore*/
    if (!check_relay_message(relay_ind, mtype))
        return 0;
    if (relay_ind)
        ctx->need_relay = 1;

    /*check the fragment relevant field
     *if fragment, insert to fragment queue and try to reassembly
     */
    if(last_fragment_ind == 0 || fid != 0)
    {
        /*because of receiving fragment cmdu, we need to record the total
         *tlvs length of message.
         *we use the received length reported by socket and to minus the
         *ether header length & cmdu message header length
         */
        temp_tlv_len = len - ETH_HLEN - sizeof(cmdu_message_header);
        debug(DEBUG_TRACE, "receive a fragment CMDU\n");
        if(last_fragment_ind == 0)
            tlv_len = calculate_fragment_tlvs_len(tlv_start, temp_tlv_len);
        else
            tlv_len = temp_tlv_len;

        insert_fragment_queue(mtype, mid, fid, last_fragment_ind, tlv_len,
                              tlv_start);
        queue_len = reassembly_fragment_queue(rx_buffer, mtype, mid);

        /*do not parse this message unless each fragment was received*/
        if (queue_len == 0) {
		/*relay the whole packets instesd of fragments to avoid dual backhaul looping*/
			if (relay_ind)
            	ctx->need_relay = 0;
        	return 0;
        }
		/*do relay in msg handle*/
		ctx->need_relay = 0;
    }

	/*because relay multicast frame will change SA to TA ensure send by neighbor
	 * so its mid need check later
	 * in order to keep reliable for relay multicast message, these message will be
	 * transfered to unicast and sent again. its relay bit will be set to zero.
	 * for dual backhaul link, discovery message cannot drop so no need check mid
	 */
	if (relay_ind == 0 && mtype != TOPOLOGY_DISCOVERY &&
		mtype != TOPOLOGY_NOTIFICATION &&
		mtype != AP_AUTOCONFIG_SEARCH &&
		mtype != AP_AUTOCONFIG_RENEW) {
		if (0 > check_recv_mid(ctx, al_mac, mid, mtype))
			return 0;
	}

    /*shift to payload, if queue_len > 0 ==> get message tlv from fragment
     *queue. otherwise, get payload from original allocated buffer
     */
    if(queue_len > 0)
        temp_buf = rx_buffer;
    else
        temp_buf = tlv_start;
	/*check if receive 1905_ack for those message which need receive*/
	if (mtype == Ack_1905)
		delete_retry_message_queue(al_mac, mid,
			(unsigned int)ctx->recent_cmdu_rx_if_number);

    switch(mtype) {
    case TOPOLOGY_DISCOVERY:
        /* need to do ==> check al mac addr tlv
         * & MID to know whether this message is duplicated
         */
        if (0 > parse_topology_discovery_message(ctx, temp_buf,
        		&need_query, &need_notify, &need_discovery,
        		al_mac, itf_mac, smac))
            break;
		if (need_discovery)
			insert_cmdu_txq((unsigned char*)p1905_multicast_address,
				ctx->p1905_al_mac_addr, e_topology_discovery, ++ctx->mid,
				ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		topology_discovery_action(ctx, al_mac, need_query, need_notify);
		update_topology_tree(ctx);
		check_neighbor_discovery(ctx, al_mac, itf_mac);
    	break;
    case TOPOLOGY_NOTIFICATION:
	{
		unsigned char sta_mac[ETH_ALEN] = {0};
		unsigned char neth_almac[ETH_ALEN] = {0};
		unsigned char bssid[ETH_ALEN] = {0};
		unsigned char event = 0;
		unsigned char zero_mac[ETH_ALEN] = {0};

        if (0 > parse_topology_notification_message(ctx, temp_buf, al_mac,
				bssid, sta_mac, &event, neth_almac)) {
			ctx->need_relay = 0;
            break;
        }
		/*need check mid for SA*/
		if (0 > check_recv_mid(ctx, al_mac, mid, mtype)) {
			ctx->need_relay = 0;
			break;
		}
		if (ctx->Certification == 0)
		{
			_1905_notify_topology_notification_event(ctx, al_mac);
		}
        /*send a topology query after receiving the topology notification message*/
        debug(DEBUG_TRACE, "respond topology query to %02x:%02x:%02x:%02x:%02x:%02x for notification\n",
			PRINT_MAC(al_mac));
        insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_topology_query, ++ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
#if 0
#ifdef SUPPORT_CMDU_RELIABLE
		SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry) {
			if(!os_memcmp(tpgr_db->al_mac_addr, al_mac, 6))
				break;
		}
		if (!tpgr_db) {
			debug(DEBUG_TRACE, "send notification from %02x:%02x:%02x:%02x:%02x:%02x for notification\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_topology_notification, ++ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		}
#endif
#endif
		if (os_memcmp(neth_almac, zero_mac, ETH_ALEN))
			detect_neighbor_existence(ctx, neth_almac, NULL);
		else if (os_memcmp(sta_mac, zero_mac, ETH_ALEN) &&
				os_memcmp(bssid, zero_mac, ETH_ALEN) && event)
			detect_neighbor_existence(ctx, NULL, sta_mac);
		else
			debug(DEBUG_TRACE, "unhandle case!\n");
    }
    	break;
    case TOPOLOGY_QUERY:
    	debug(DEBUG_TRACE, "send topology response to %02x:%02x:%02x:%02x:%02x:%02x for query\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_topology_response, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
#if 0
		if (0> find_al_mac_address(ctx, smac, al_mac)) {
			debug(DEBUG_ERROR, "cannot find al mac(%02x:%02x:%02x:%02x:%02x:%02x) in database, query it\n", PRINT_MAC(smac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_topology_query, ++ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		}
#endif
#ifdef MAP_R2
		if(0 > parse_topology_query_message(ctx,temp_buf,al_mac)) {
			debug(DEBUG_ERROR, "receive error topology response message\n");
			break;
		}
#endif
    	break;
	case TOPOLOGY_RESPONSE:
		if(0 > parse_topology_response_message(ctx,temp_buf,al_mac)) {
			debug(DEBUG_ERROR, "receive error topology response message\n");
			break;
		}
		if(ctx->Certification == 0)
		{
			_1905_notify_topology_rsp_event(ctx, al_mac, ctx->itf[ctx->recent_cmdu_rx_if_number].mac_addr);
		}
		query_neighbor_info(ctx, al_mac, ctx->recent_cmdu_rx_if_number);
		update_topology_tree(ctx);
		break;
    case VENDOR_SPECIFIC:
        if(0 > parse_vendor_specific_message(ctx, temp_buf, &vs_info, &internal_vendor_message))
        {
            debug(DEBUG_TRACE, "receive other vendor's vendor specific message\n");
            break;
        }
		if (ctx->Certification == 0) {
			/*some vendor message is internal used by 1905daemon, so do not notify it to upper layer*/
			if (!internal_vendor_message)
				_1905_notify_vendor_specific_message(ctx, al_mac);
			else {
				debug(DEBUG_OFF, "handle 1905 internal using vendor specific message\n");
				parse_1905_internal_vendor_specific_message(ctx, temp_buf);
			}
		}

		if (relay_ind && queue_len) {
			relay_fragment_send(ctx, rx_buffer, queue_len,
				(buf - ETH_HLEN), ctx->p1905_al_mac_addr);
		}
    	break;
    case LINK_METRICS_QUERY:
        if(0 > parse_link_metric_query_message(ctx, temp_buf))
        {
            debug(DEBUG_ERROR, "receive error LINK_METRICS_QUERY message\n");
            break;
        }
		if(ctx->Certification == 0)
		{
			if(_1905_notify_link_metrics_query(ctx, al_mac) < 0)
			{
				break;
			}
		}
        insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_link_metric_response, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		break;
    case AP_AUTOCONFIG_SEARCH:
        if(0 > parse_ap_autoconfig_search_message(ctx, temp_buf, al_mac))
        {
            debug(DEBUG_ERROR, "no need to response this autoconfig search message\n");
			ctx->need_relay = 0;
            break;
        }

		if (0 > check_recv_mid(ctx, al_mac, mid, mtype)) {
			ctx->need_relay = 0;
			break;
		}
		if (ctx->role == CONTROLLER) {
			/*insert agent entry*/
			insert_agent_info(ctx, al_mac);
	        /*send ap auto config response message, unicast*/
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_ap_autoconfiguration_response, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		}
		break;
    case AP_AUTOCONFIG_RESPONSE:
//by zheng.zhou  set to 0 to resolve issue.
		ctx->is_authenticator_exist_in_M2 = 0;
		debug(DEBUG_TRACE, "set ctx->is_authenticator_exist_in_M2 = 0\n");
		if(ctx->autoconfig_search_mid != mid)
		{
			debug(DEBUG_ERROR, "mid mismatch\n");
			break;
		}

        if(0 > parse_ap_autoconfig_response_message(ctx, temp_buf, &band, &service, &role, al_mac))
        {
            debug(DEBUG_ERROR, "no need to response this autoconfig response message\n");
            break;
        }

		memcpy(ctx->cinfo.almac, al_mac, ETH_ALEN);
		ctx->cinfo.supported_freq = band;
		ctx->cinfo.supported_role_registrar = role;
		memcpy(ctx->cinfo.local_ifname, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, IFNAMSIZ);
		ctx->cinfo.recv_ifid = ctx->recent_cmdu_rx_if_number;
		debug(DEBUG_TRACE, "find the conroller with supperted_freq=%d, role=%d\n", band, role);
		debug(DEBUG_TRACE, "recv conroller with interface=%s, ifidx=%d\n",
			 ctx->cinfo.local_ifname, ctx->cinfo.recv_ifid);
		debug(DEBUG_OFF, "find the conroller al_mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));

		if(ctx->enrolle_state == no_ap_autoconfig && ctx->controller_search_state == controller_search_idle)
		{
			_1905_notify_autoconfig_rsp_event(ctx, al_mac);
			break;
		}
        if(((ctx->enrolle_state != wait_4_recv_ap_autoconfig_resp) &&
            ctx->controller_search_state == controller_search_idle)) {
           debug(DEBUG_ERROR, "enrolle_state mismatch\n");
           break;
        }

		if(ctx->controller_search_state != controller_search_idle)
		{
			debug(DEBUG_ERROR, "recv the autoconfig search rsp, notify the wapp search done\n");
			ctx->controller_search_state = controller_search_done;
			eloop_cancel_timeout(ap_controller_search_step, (void *)ctx, NULL);
			eloop_register_timeout(0, 0, ap_controller_search_step, (void *)ctx, NULL);
			break;
		}
        break;

    case AP_AUTOCONFIG_RENEW:
		{
			ctx->is_renew_ongoing = 1;
			if(0 > parse_ap_autoconfig_renew_message(ctx, temp_buf, al_mac, &band))
			{
				debug(DEBUG_ERROR, "no need to response this autoconfig renew message\n");
				ctx->need_relay = 0;
	            break;
	        }

			if (0 > check_recv_mid(ctx, al_mac, mid, mtype)) {
				ctx->need_relay = 0;
				break;
			}
			debug(DEBUG_ERROR, "recv renew message for band %s from %02x:%02x:%02x:%02x:%02x:%02x\n",
				band ? "5G" : "24G", PRINT_MAC(al_mac));
			memcpy(ctx->cinfo.almac, al_mac, ETH_ALEN);
			ctx->cinfo.supported_freq = band;
			memcpy(ctx->cinfo.local_ifname, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, IFNAMSIZ);
			ctx->cinfo.recv_ifid = ctx->recent_cmdu_rx_if_number;
			debug(DEBUG_TRACE, "find the conroller with supperted_freq=%d\n", band);
			debug(DEBUG_TRACE, "recv conroller with interface=%s, ifidx=%d\n",
				 ctx->cinfo.local_ifname, ctx->cinfo.recv_ifid);

			set_radio_autoconf_prepare(ctx, band, 1);
			triger_autoconfiguration(ctx);
    	}
        break;
    case AP_AUTOCONFIG_WSC:
		if (ctx->role == AGENT) {
	        if(ctx->enrolle_state != wait_4_recv_m2)
	            break;
	        debug(DEBUG_OFF, "got WSC M2\n");
			//hex_dump("WSC M2", (buf - ETH_HLEN), len);
		}

        if (0 > parse_ap_autoconfig_wsc_message(ctx, al_mac, temp_buf, &radio)) {
            debug(DEBUG_ERROR, "receive error AP_AUTOCONFIG_WSC message\n");
            break;
        }

		if (ctx->role == AGENT) {
#ifdef MAP_R2
			eloop_register_timeout(15, 0, map_r2_notify_ts_config, (void *)ctx, NULL);
#endif
	        /*we got correct M2, so enter the set config state*/
	        ctx->enrolle_state = wait_4_set_config;
			debug(DEBUG_TRACE, "enter in wait_4_set_config sm\n");
			eloop_cancel_timeout(ap_autoconfig_enrolle_step, (void *)ctx, NULL);
			eloop_register_timeout(0, 0, ap_autoconfig_enrolle_step, (void *)ctx, NULL);

			if (ctx->is_renew_ongoing) {
				/*send private m2 ack for agent*/
				insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
					e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
				debug(DEBUG_TRACE, "respond ack(%04x) for %02x:%02x:%02x:%02x:%02x:%02x\n",
					mid, PRINT_MAC(al_mac));
				eloop_cancel_timeout(agent_apply_new_config, (void *)ctx, NULL);
				debug(DEBUG_TRACE, "radio_cap(%02x) cur_conf_radio(%02x)\n",
					ctx->radio_cap, ctx->cur_conf_radio);
				switch (ctx->cur_conf_radio) {
				case RADIO_2G_CAP:
					if ((ctx->radio_cap & ~RADIO_2G_CAP) == (RADIO_5GL_CAP | RADIO_5GH_CAP)) {
						eloop_register_timeout(TWO_RADIO_CONFIG_TIME_MAX, 0,
							agent_apply_new_config, (void *)ctx, NULL);
					} else if ((ctx->radio_cap & ~RADIO_2G_CAP) &
									(RADIO_5GL_CAP | RADIO_5GH_CAP | RADIO_5G_CAP)) {
						eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
							agent_apply_new_config, (void *)ctx, NULL);
					} else {
						eloop_register_timeout(0, 0,
							agent_apply_new_config, (void *)ctx, NULL);
					}
				break;
				case RADIO_5GL_CAP:
					if (ctx->radio_cap & RADIO_5GH_CAP)
						eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
								agent_apply_new_config, (void *)ctx, NULL);
					else
						eloop_register_timeout(0, 0,
							agent_apply_new_config, (void *)ctx, NULL);
				break;
				case RADIO_5GH_CAP:
				case RADIO_5G_CAP:
					eloop_register_timeout(0, 0,
							agent_apply_new_config, (void *)ctx, NULL);
				break;
				default:
					debug(DEBUG_ERROR, "error radio_cap(%02x)\n", ctx->radio_cap);
					eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
							agent_apply_new_config, (void *)ctx, NULL);
				break;
				}
			}
		} else {
			debug(DEBUG_OFF, "got WSC M1\n");
			//hex_dump("WSC M1", (buf - ETH_HLEN), len);

			/* Registrar got correct M1. Need to response M2.(Unicast)
			 * It has a strange behavior about mid. Spec does not specify we
			 * need to use same mid for M1 and M2. So I create a new mid for M2
			 */
			debug(DEBUG_TRACE, "send WSC M2 to %02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_ap_autoconfiguration_wsc_m2, ++ctx->mid,
				ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if (ctx->role == CONTROLLER && ctx->trigger_renew) {
				leaf_info *leaf_top = NULL;
				int ret = 0;

				ret = get_top(&ctx->topo_stack, &leaf_top);
				if (ret < 0) {
					debug(DEBUG_ERROR, "BUG here!!!! get top node from stack error\n");
					break;
				}
				if (!os_memcmp(leaf_top->al_mac, al_mac, ETH_ALEN)) {
					debug(DEBUG_TRACE, "current_renew_almac %02x:%02x:%02x:%02x:%02x:%02x, "
						"current_renew_band: %02x, peer_bss_rf_band = %02x\n",
					PRINT_MAC(leaf_top->al_mac), leaf_top->cur_renew_band, ctx->peer_bss_rf_band);
					eloop_cancel_timeout(controller_renew_bss_step, (void *)ctx, NULL);
					leaf_top->cur_renew_radio = radio;
					leaf_top->m2_ack_mid = ctx->mid;
					ctx->renew_state = wait_4_recv_m2_ack;
					if (radio == RADIO_2G_CAP) {
						eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
							controller_renew_bss_step, (void *)ctx, NULL);
					} else if (radio == RADIO_5GL_CAP || radio == RADIO_5GH_CAP) {
						if ((leaf_top->fin_renew_radio & ~RADIO_2G_CAP) & (RADIO_5GL_CAP | RADIO_5GH_CAP)) {
							eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
								controller_renew_bss_step, (void *)ctx, NULL);
						} else {
							if ((leaf_top->radio_cap & ~RADIO_2G_CAP) == (RADIO_5GL_CAP | RADIO_5GH_CAP))
								eloop_register_timeout(TWO_RADIO_CONFIG_TIME_MAX, 0,
									controller_renew_bss_step, (void *)ctx, NULL);
							else
								eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
									controller_renew_bss_step, (void *)ctx, NULL);
						}
					} else {
						eloop_register_timeout(ONE_RADIO_CONFIG_TIME_MAX, 0,
							controller_renew_bss_step, (void *)ctx, NULL);
					}

					debug(DEBUG_TRACE, "m2_ack_mid(%04x) radio(%02x) fin_renew_radio(%02x)\n",
						leaf_top->m2_ack_mid, radio, leaf_top->fin_renew_radio);
				}
			}
		}
    	break;
	case Ack_1905:
	{
		leaf_info *leaf_top = NULL;
		int ret = 0;

		if (ctx->role == CONTROLLER && ctx->trigger_renew) {
			ret = get_top(&ctx->topo_stack, &leaf_top);
			if (ret < 0) {
				debug(DEBUG_ERROR, "BUG here!!!! get top node from stack error\n");
				break;
			}

			if (leaf_top->m2_ack_mid == mid) {
				debug(DEBUG_TRACE, "m2_ack_mid(%04x) mid(%04x) almac %02x:%02x:%02x:%02x:%02x:%02x\n",
					leaf_top->m2_ack_mid, mid, PRINT_MAC(leaf_top->al_mac));
				debug(DEBUG_TRACE, "band_cap(%02x) radio_cap(%02x) cur_renew_band(%02x)"
					" cur_renew_radio(%02x) renew_retry_cnt(%d)\n",
					leaf_top->band_cap, leaf_top->radio_cap, leaf_top->cur_renew_band,
					leaf_top->cur_renew_radio, leaf_top->renew_retry_cnt);
				leaf_top->fin_renew_radio |= leaf_top->cur_renew_radio;
				if (leaf_top->cur_renew_radio == RADIO_2G_CAP) {
					eloop_cancel_timeout(controller_renew_bss_step, (void *)ctx, NULL);
					if (leaf_top->band_cap & (BAND_5GL_CAP | BAND_5GH_CAP | BAND_5G_CAP)) {
						leaf_top->renew_retry_cnt = 3;
						leaf_top->cur_renew_band = 1;
						ctx->renew_state = wait_4_send_renew;
					} else {
						pop_node(&ctx->topo_stack);
						ctx->renew_state = wait_4_pop_node;
					}
					eloop_register_timeout(0, 0, controller_renew_bss_step, (void *)ctx, NULL);
				} else if (leaf_top->cur_renew_radio == RADIO_5GL_CAP || leaf_top->cur_renew_radio == RADIO_5GH_CAP) {
					if ((leaf_top->fin_renew_radio & ~RADIO_2G_CAP) == (RADIO_5GL_CAP | RADIO_5GH_CAP)) {
						eloop_cancel_timeout(controller_renew_bss_step, (void *)ctx, NULL);
						pop_node(&ctx->topo_stack);
						ctx->renew_state = wait_4_pop_node;
						eloop_register_timeout(0, 0, controller_renew_bss_step, (void *)ctx, NULL);
					} else {
						if (!(leaf_top->radio_cap & RADIO_5GL_CAP) || !(leaf_top->radio_cap & RADIO_5GH_CAP)) {
							eloop_cancel_timeout(controller_renew_bss_step, (void *)ctx, NULL);
							pop_node(&ctx->topo_stack);
							ctx->renew_state = wait_4_pop_node;
							eloop_register_timeout(0, 0, controller_renew_bss_step, (void *)ctx, NULL);
						}
					}
				} else {
					eloop_cancel_timeout(controller_renew_bss_step, (void *)ctx, NULL);
					pop_node(&ctx->topo_stack);
					ctx->renew_state = wait_4_pop_node;
					eloop_register_timeout(0, 0, controller_renew_bss_step, (void *)ctx, NULL);
				}
			}
		}
	}
		break;
	case AP_CAPABILITY_QUERY:
    	debug(DEBUG_TRACE, "send ap_capability_report to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_ap_capability_report,
			mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		break;
	case AP_CAPABILITY_REPORT:
		debug(DEBUG_TRACE, "receive ap_capability_report from %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		if (0 > parse_ap_capability_report_message(ctx, al_mac, temp_buf)) {
			debug(DEBUG_ERROR, "receive error ap capability report message\n");
			break;
		}
		if (ctx->Certification == 0) {
			_1905_notify_ap_capability_report_event(ctx, al_mac);
		}
		break;
	case CLIENT_CAPABILITY_QUERY:
		if(0 > parse_client_capability_query_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "receive error client capability query message, no need response\n");
			break;
		}
    	debug(DEBUG_TRACE, "send client_capability_report to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_cli_capability_report,
			mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		break;
	case CLIENT_CAPABILITY_REPORT:
		debug(DEBUG_TRACE, "receive client capability report from %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		if (0 > parse_client_capability_report_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "receive error client capability report message\n");
			break;
		}
		if (ctx->Certification == 0) {
			_1905_notify_client_capability_report_event(ctx, al_mac);
		}
		break;
	case CHANNLE_PREFERENCE_QUERY:
	{
		if(ctx->Certification == 1)
		{
			wapp_get_channel_repinfo(ctx);
		}
		else
		{
			ctx->rcv_chprefer_query = 1;
			reset_stored_tlv(ctx);
			if(_1905_notify_channel_preference_query(ctx, al_mac) < 0)
			{
				debug(DEBUG_ERROR, "get channel prefence fail\n");
				ctx->rcv_chprefer_query = 0;
				break;
			}
		}
		debug(DEBUG_TRACE, "send channel preference report to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_channel_preference_report,
			mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		ctx->rcv_chprefer_query = 0;
	}
		break;
	case CHANNLE_SELECTION_REQUEST:
	{
		if(ctx->Certification == 1)
		{
			/*delete all the stored channel preference information*/
			delete_all_ch_prefer_info(&ctx->ap_cap_entry.ch_prefer_head);
		}
		/*parse the selection request message*/
		if(0 > parse_channel_selection_request_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this channel selection request message\n");
            break;
        }
		if(ctx->Certification == 1)
		{
			/*check if need change channel setting*/
			update_channel_setting(ctx, al_mac, mid);
		}
		else
		{
			if(_1905_notify_channel_selection_req(ctx, al_mac) < 0)
			{
				break;
			}
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_channel_selection_response,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		}
	}
		break;
	case OPERATING_CHANNEL_REPORT:
		debug(DEBUG_TRACE, "receive operating channel report\n");
   		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		if(0 > parse_operating_channel_report_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "receive error operating channel report message\n");
            break;
        }
		if (ctx->Certification == 0) {
			_1905_notify_operating_channel_report_event(ctx, al_mac);
		}
		break;
	case e_higher_layer_data:
		debug(DEBUG_TRACE, "receive higher layer data\n");
   		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		if(0 > parse_higher_layer_data_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "receive error operating channel report message\n");
            break;
        }
		if (ctx->Certification == 0) {
			_1905_notify_higher_layer_data_event(ctx, al_mac);
		}
		break;
	case CLIENT_STEERING_REQUEST:
	{
		debug(DEBUG_TRACE, "receive client steering request\n");
   		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
    	insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse the selection request message*/
		if(0 > parse_client_steering_request_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this client steering request message\n");
            break;
        }
		if(ctx->Certification == 1)
		{
			wapp_set_info_by_msgtype(ctx, WAPP_USER_SET_STEERING_SETTING, NULL,
				(void *)ctx->cli_steer_req, ctx->steer_req_len);
			free(ctx->cli_steer_req);
			ctx->cli_steer_req = NULL;
			ctx->steer_req_len = 0;
		}
		else
		{
			_1905_set_steering_setting(ctx, al_mac);
		}
	}
		break;
	case MAP_POLICY_CONFIG_REQUEST:
	{
		if (ctx->role == AGENT) {
			unsigned char old_report_interval = ctx->map_policy.mpolicy.report_interval;

			debug(DEBUG_TRACE, "receive map policy config request\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
				e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
			/*parse the map policy config request message*/
			if(0 > parse_map_policy_config_request_message(ctx, temp_buf)) {
				debug(DEBUG_ERROR, "error! no need to response this map policy config request message\n");
				break;
			}

			if(ctx->Certification == 0)
			{
				_1905_set_map_policy_config(ctx, al_mac);
			}

			if (ctx->map_policy.mpolicy.report_interval == 0) {
				eloop_cancel_timeout(metrics_report_timeout, (void *)ctx, NULL);
			} else if (old_report_interval != ctx->map_policy.mpolicy.report_interval) {
				eloop_cancel_timeout(metrics_report_timeout, (void *)ctx, NULL);
				eloop_register_timeout(ctx->map_policy.mpolicy.report_interval, 0, metrics_report_timeout, (void *)ctx, NULL);
			}
#ifdef MAP_R2
			eloop_register_timeout(4, 0, map_r2_notify_ts_config, (void *)ctx, NULL);
#endif
		}
	}
		break;
	case CLIENT_ASSOC_CONTROL_REQUEST:
	{
		int datalen = 0;

		debug(DEBUG_TRACE, "receive client assoc control request\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse the client association control request message*/
		if(0 > parse_cli_assoc_control_request_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to response this client association control request message\n");
			break;
		}

		if(ctx->Certification == 1)
		{
			/*send the client association control request info to wapp*/
			datalen = sizeof(struct control_policy) + ctx->steer_cntrl->sta_list_count * ETH_ALEN;
			wapp_set_info_by_msgtype(ctx, WAPP_USER_SET_ASSOC_CNTRL_SETTING, NULL,
				(void *)ctx->steer_cntrl, datalen);
			free(ctx->steer_cntrl);
			ctx->steer_cntrl = NULL;
		}
		else
		{
			_1905_set_client_assoc_ctrl(ctx, al_mac);
		}
	}
		break;
	/*link metric collection*/
	case AP_LINK_METRICS_QUERY:
	{
		/*parse AP_LINK_METRICS_QUERY msg*/

		if (0 > parse_ap_metrics_query_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this link metrics query message\n");
            break;
        }

		if(ctx->Certification == 1)
		{
			/*query ap metrics response info from wapp*/
			wapp_get_ap_metrics_info(ctx);
			wapp_get_assoc_sta_traffic_stats(ctx);
			wapp_get_all_assoc_sta_link_metrics(ctx);
		}
		else
		{
			ctx->rcv_apmetrics_query = 1;
			if(_1905_notify_ap_metrics_query(ctx, al_mac
#ifdef MAP_R2
			, 0
#endif
			) < 0)
			{
				debug(DEBUG_ERROR, "notify ap metrics query fail\n");
				ctx->rcv_apmetrics_query = 0;
				break;
			}
		}

		debug(DEBUG_TRACE, "send ap metrics response to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_ap_metrics_response, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		ctx->rcv_apmetrics_query = 0;

	}
		break;
	case AP_LINK_METRICS_RESPONSE:
	{
		if (ctx->role == CONTROLLER) {
			struct agent_list_db *agent_info = NULL;
			if (ctx->Certification == 1) {
				debug(DEBUG_TRACE, "got AP_LINK_METRICS_RESPONSE\n");
				find_agent_info(ctx, al_mac, &agent_info);
				if (!agent_info) {
					debug(DEBUG_ERROR, "error! no agent info exist\n");
					break;
				}
				/*delete all the stored metrics info for the agent*/
				delete_agent_ap_metrics_info(&agent_info->metrics_rsp_head);
			}
			/*parse ap metrics response message for the agent*/
			if(0 > parse_ap_metrics_response_message(ctx, agent_info, temp_buf)) {
				debug(DEBUG_ERROR, "error! parse ap metrics response message\n");
			}
			if (ctx->Certification == 0) {
				_1905_notify_ap_metrics_response(ctx, al_mac);
			}
		}
	}
		break;
	case LINK_METRICS_RESPONSE:
	{
		struct agent_list_db *agent_info = NULL;

		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "got LINK_METRICS_RESPONSE\n");
			if(ctx->Certification == 1)
			{
				find_agent_info(ctx, al_mac, &agent_info);
				if (!agent_info) {
					debug(DEBUG_ERROR, "error! no agent info exist\n");
					break;
				}
				/*delete all the stored link metrics info for the agent*/
				delete_agent_tx_link_metrics_info(&agent_info->tx_metrics_head);
				delete_agent_rx_link_metrics_info(&agent_info->rx_metrics_head);
			}
			/*parse ap metrics response message for the agent*/
			if(0 > parse_link_metrics_response_message(ctx, agent_info, temp_buf)) {
				debug(DEBUG_ERROR, "error! parse link metrics response message\n");
			}
			if(ctx->Certification == 0)
			{
				_1905_notify_link_metrics_response(ctx, al_mac);
			}
		}
	}
		break;
	case ASSOC_STA_LINK_METRICS_QUERY:
	{
		/*parse AP_LINK_METRICS_QUERY msg*/
		if(0 > parse_associated_sta_link_metrics_query_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to response this associated sta link metrics query message\n");
			break;
		}

		if(ctx->Certification == 1)
		{
			/*query ap metrics response info from wapp*/
			if (0 > wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_ONE_ASSOC_STA_LINK_METRICS,
						LIB_ONE_ASSOC_STA_LINK_METRICS, NULL, ctx->metric_entry.assoc_sta, NULL, 0)) {
				debug(DEBUG_ERROR, "error! wapp get one assoc sta link metrics\n");
				break;
			}
		}
		else
		{
			if(_1905_notify_assoc_sta_link_metrics_query(ctx, al_mac) < 0)
			{
				debug(DEBUG_ERROR, "error! wapp get one assoc sta link metrics\n");
				break;
			}
		}
		debug(DEBUG_TRACE, "send associated sta link metrics response to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_associated_sta_link_metrics_response,
			mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
	}
		break;
	case ASSOC_STA_LINK_METRICS_RESPONSE:
	{
		if(parse_associated_sta_link_metrics_rsp_message(ctx, temp_buf) < 0)
		{
			debug(DEBUG_ERROR, "parse associated sta link metrics query message error\n");
			break;
		}
		if(ctx->Certification == 0)
		{
			_1905_notify_assoc_sta_link_metric_rsp(ctx, al_mac);
		}
	}
		break;
	case UNASSOC_STA_LINK_METRICS_QUERY:
	{
		int query_len = 0;

		debug(DEBUG_TRACE, "receive unassoc sta link metrics query\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse AP_LINK_METRICS_QUERY msg*/
		if(0 > parse_unassociated_sta_link_metrics_query_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to response this unassociated sta link metrics query message\n");
			break;
		}

		if(ctx->Certification == 1)
		{
			/*query ap metrics response info from wapp*/
			query_len = sizeof(struct unlink_metrics_query) +
				(ctx->metric_entry.unlink_query)->sta_num * ETH_ALEN;
			if (0 > wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_UNASSOC_STA_LINK_METRICS,
						LIB_UNASSOC_STA_LINK_METRICS, NULL, NULL,
						(void *)ctx->metric_entry.unlink_query, query_len)) {
				debug(DEBUG_ERROR, "error! wapp_get_unassoc_sta_link_metrics\n");
				free(ctx->metric_entry.unlink_query);
				ctx->metric_entry.unlink_query = NULL;
				break;
			}
			free(ctx->metric_entry.unlink_query);
			ctx->metric_entry.unlink_query = NULL;
		}
		else
		{
			if(_1905_notify_unassoc_sta_metrics_query(ctx, al_mac) < 0)
				break;
		}
		debug(DEBUG_TRACE, "send unassociated sta link metrics response to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		ctx->mid++;
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_unassociated_sta_link_metrics_response,
			ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
	}
		break;
	case UNASSOC_STA_LINK_METRICS_RESPONSE:
	{
		debug(DEBUG_TRACE, "receive unassoc sta link metrics response\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		if(0 > parse_unassoc_sta_link_metrics_response_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! unassoc sta link metrics response message\n");
			break;
		}

		if(ctx->Certification == 0)
		{
			_1905_notify_unassoc_sta_link_metric_rsp(ctx, al_mac);
		}
	}
		break;
	case BEACON_METRICS_QUERY:
	{
		int datalen = 0;

		debug(DEBUG_TRACE, "receive beacon metrics query\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse AP_LINK_METRICS_QUERY msg*/
		if(0 > parse_beacon_metrics_query_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to response this beacon metrics query message\n");
			break;
		}

		if(ctx->Certification == 1)
		{
		/*query ap metrics response info from wapp*/
			datalen = sizeof(struct beacon_metrics_query) +
						ctx->metric_entry.bcn_query->ap_ch_rpt_num * sizeof(struct ap_chn_rpt);
			if (0 > wapp_set_info_by_msgtype(ctx, WAPP_USER_SET_BEACON_METRICS_QRY, NULL,
						(void *)ctx->metric_entry.bcn_query, datalen)) {
				debug(DEBUG_ERROR, "error! WAPP_USER_SET_BEACON_METRICS_QRY\n");
				free(ctx->metric_entry.bcn_query);
				ctx->metric_entry.bcn_query = NULL;
				break;
			}
			free(ctx->metric_entry.bcn_query);
			ctx->metric_entry.bcn_query = NULL;
		}
		else
		{
			_1905_notify_beacon_metrics_query(ctx, al_mac);
		}
	}
		break;
	case BEACON_METRICS_RESPONSE:
	{
		debug(DEBUG_TRACE, "receive beacon metrics response\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		if(0 > parse_beacon_metrics_response_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! beacon metrics response message\n");
			break;
		}

		if(ctx->Certification == 0)
		{
			_1905_notify_bcn_metric_rsp(ctx, al_mac);
		}
	}
		break;
	case CHANNLE_PREFERENCE_REPORT:
	{
		struct agent_list_db *agent_info = NULL;

		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "receive channel preference report\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr, e_1905_ack,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if(ctx->Certification == 1)
			{
				find_agent_info(ctx, smac, &agent_info);
				if (!agent_info) {
					debug(DEBUG_ERROR, "error! no agent info exist\n");
					break;
				}
				/*delete all the stored channel preference information for the agent*/
				delete_agent_ch_prefer_info(ctx, agent_info);
			}
			/*parse the channel preference report message for the agent*/
			if(0 > parse_channel_preference_report_message(ctx, agent_info, temp_buf)) {
				debug(DEBUG_ERROR, "error! parse channel preference report message\n");
				break;
			}
			if(ctx->Certification == 0)
			{
				_1905_notify_channel_preference_report(ctx, al_mac);
			}
		}
	}
		break;
	case CHANNLE_SELECTION_RESPONSE:
	{
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "receive channel selection response\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr, e_1905_ack,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if (0 > parse_channel_selection_rsp_message(ctx, temp_buf)) {
				debug(DEBUG_TRACE, "error! parse channel selection response message\n");
				break;
			}
			if (ctx->Certification == 0) {
				_1905_notify_ch_selection_rsp_event(ctx, al_mac);
			}
		}
	}
		break;
	case CLIENT_STEERING_BTM_REPORT:
	{
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "receive client steering btm report\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr, e_1905_ack,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if(0 > parse_client_steering_btm_report_message(ctx, temp_buf)) {
				debug(DEBUG_ERROR, "error! parse client steering btm report message\n");
				break;
			}

			if (ctx->Certification == 0) {
				_1905_notify_client_steering_btm_report(ctx, al_mac);
			}
		}
	}
		break;
	case CLIENT_STEERING_COMPLETED:
	{
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "receive client steering completed\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr, e_1905_ack,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if(ctx->Certification == 0)
			{
				reset_stored_tlv(ctx);
				_1905_notify_steering_complete(ctx, al_mac);
			}
		}

	}
		break;
	case BACKHAUL_STEERING_RESPONSE:
	{
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "receive backhaul steering response\n");
			debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(al_mac));
			insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr, e_1905_ack,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

			if(parse_backhaul_steering_rsp_message(ctx, temp_buf) < 0)
			{
				debug(DEBUG_ERROR, "error! parse backhual steering response message\n");
	            break;
			}
			if(ctx->Certification == 0)
			{
				_1905_notify_bh_steering_rsp(ctx, al_mac);
			}
		}
	}
		break;
	/*Backhaul optimization*/
	case BACKHAUL_STEERING_REQUEST:
	{
		debug(DEBUG_TRACE, "receive backhaul steering request\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse BACKHAUL_STEERING_REQUEST msg*/
		if(0 > parse_backhaul_steering_request_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this backhaul steering request message\n");
            break;
        }
		if (ctx->Certification == 1) {
			if (check_invalid_channel(ctx->bsteer_req.oper_class, 1, &ctx->bsteer_req.channel)) {
				wapp_set_info_by_msgtype(ctx, WAPP_USER_SET_BACKHAUL_STEER, ctx->bsteer_req.backhaul_mac,
						(void *)&ctx->bsteer_req, sizeof(ctx->bsteer_req));
			} else {
				debug(DEBUG_TRACE, "invalid channel or opclass\n");
				memcpy(ctx->bsteer_rsp.backhaul_mac, ctx->bsteer_req.backhaul_mac, ETH_ALEN);
				memcpy(ctx->bsteer_rsp.target_bssid, ctx->bsteer_req.target_bssid, ETH_ALEN);
				ctx->bsteer_rsp.status = 0x01;
				debug(DEBUG_TRACE, "send backhaul_steering_response to %02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(al_mac));
				insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
					e_backhaul_steering_response, ++ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
			}
		} else {
			_1905_notify_bh_steering_req(ctx, al_mac);
		}
	}
		break;
	case COMBINED_INFRASTRUCTURE_METRICS:
	{
		debug(DEBUG_TRACE, "receive combined infrastructure metrics\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		if(0 > parse_combined_infra_metrics_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this backhaul steering request message\n");
            break;
        }
		if(ctx->Certification == 0)
		{
			_1905_notify_combined_infra_metrics(ctx, al_mac);
		}
	}
		break;
#ifdef MAP_R2
	/*channel scan feature*/
	case CHANNEL_SCAN_REQUEST:
	{
		debug(DEBUG_TRACE, "receive channel scan request message\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse BACKHAUL_STEERING_REQUEST msg*/
		if(0 > parse_channel_scan_request_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this channel scan request message\n");
            break;
        }

		_1905_notify_ch_scan_req(ctx, al_mac);
	}
		break;
	case CHANNEL_SCAN_REPORT:
	{
		debug(DEBUG_TRACE, "receive channel scan report message\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse BACKHAUL_STEERING_REQUEST msg*/
		if(0 > parse_channel_scan_report_message(ctx, temp_buf)) {
            debug(DEBUG_ERROR, "error! no need to response this channel scan report message\n");
            break;
        }

		_1905_notify_ch_scan_rep(ctx, al_mac);
	}
		break;

	case TUNNELED_MESSAGE:
	{
		debug(DEBUG_TRACE, "receive TUNNELED_MESSAGE\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse tunneled msg*/
		if(0 > parse_tunneled_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send tunneled msg to mapd\n");
			break;
		}

		_1905_notify_tunneled_msg(ctx, al_mac);

	}
        break;

	case ASSOCIATION_STATUS_NOTIFICATION:
	{
		debug(DEBUG_TRACE, "receive ASSOCIATION_STATUS_NOTIFICATION\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse assoc status notification msg*/
		if(0 > parse_assoc_status_notification_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send assoc status notification msg to mapd\n");
			break;
		}

		_1905_notify_assoc_status_notification_event(ctx, al_mac);

	}
        break;

        case CAC_REQUEST:
	{
		debug(DEBUG_TRACE, "receive CAC_REQUEST\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse assoc status notification msg*/
		if(0 > parse_cac_request_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send cac request msg to mapd\n");
			break;
		}

		_1905_notify_cac_request_event(ctx, al_mac);

        }
	break;

        case CAC_TERMINATION:
	{
		debug(DEBUG_TRACE, "receive CAC_TERMINATION\n");
		debug(DEBUG_TRACE, "send 1905 ack to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(al_mac));
		insert_cmdu_txq(al_mac, ctx->p1905_al_mac_addr,
			e_1905_ack, mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);

		/*parse assoc status notification msg*/
		if(0 > parse_cac_terminate_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send cac terminate msg to mapd\n");
			break;
		}

		_1905_notify_cac_terminate_event(ctx, al_mac);

        }
	break;

	case CLIENT_DISASSOCIATION_STATS:
	{
		debug(DEBUG_TRACE, "receive CLIENT_DISASSOCIATION_STATS\n");

		hex_dump_all("CLIENT_DISASSOCIATION_STATS",temp_buf, len-25);

		/*parse client disassociation stats msg*/
		if(0 > parse_client_disassciation_stats_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send client disassoctiation stats to mapd\n");
			break;
		}

		_1905_notify_client_disassociation_stats_event(ctx, al_mac);
	}
		break;

	case FAILED_ASSOCIATION_MESSAGE:
	{
		debug(DEBUG_TRACE, "receive FAILED_ASSOCIATION_MESSAGE\n");

		hex_dump_all("FAILED_ASSOCIATION_MESSAGE",temp_buf, len-25);

		/*parse failed association msg*/
		if(0 > parse_failed_association_message(ctx, temp_buf)) {
			debug(DEBUG_ERROR, "error! no need to send failed association message to mapd\n");
			break;
		}

		_1905_notify_failed_assoc_event(ctx, al_mac);
	}
#endif // #ifdef MAP_R2
	default:
		break;
	}
    return 0;
}

int insert_topology_discovery_database(struct p1905_managerd_ctx *ctx,
	unsigned char *al_mac, unsigned char *itf_mac, unsigned char *recv_itf_mac)
{
	struct topology_discovery_db *tpg_db = NULL;

    LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
    {
		if (!os_memcmp(tpg_db->al_mac, al_mac, 6) &&
			!os_memcmp(tpg_db->itf_mac, itf_mac, 6)) {
			return 0;
		}
    }

	if (!tpg_db) {
		tpg_db = (struct topology_discovery_db *)malloc(sizeof(struct topology_discovery_db));
		if (!tpg_db)
			return -1;
		os_memset(tpg_db, 0, sizeof(*tpg_db));
		os_memcpy(tpg_db->al_mac, al_mac, ETH_ALEN);
		os_memcpy(tpg_db->itf_mac, itf_mac, ETH_ALEN);
		os_memcpy(tpg_db->receive_itf_mac, recv_itf_mac, ETH_ALEN);
		tpg_db->time_to_live = TOPOLOGY_DISCOVERY_TTL;
		LIST_INSERT_HEAD(&ctx->topology_entry.tpddb_head, tpg_db, tpddb_entry);
		debug(DEBUG_OFF, "insert discovery db "
				"al_mac(%02x:%02x:%02x:%02x:%02x:%02x) "
				"itf_mac(%02x:%02x:%02x:%02x:%02x:%02x) "
				"receive_itf_mac(%02x:%02x:%02x:%02x:%02x:%02x) ",
				PRINT_MAC(al_mac), PRINT_MAC(itf_mac),
				PRINT_MAC(recv_itf_mac));
	}

	return 1;
}

struct topology_discovery_db * find_discovery_by_almac(
	struct p1905_managerd_ctx *ctx, unsigned char *al_mac)
{
	struct topology_discovery_db *tpg_db = NULL;

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
	{
		if (!os_memcmp(tpg_db->al_mac, al_mac, ETH_ALEN)) {
			return tpg_db;
		}
	}

	return tpg_db;
}

void find_neighbor_almac_by_intf_mac(
	struct p1905_managerd_ctx *ctx, unsigned char *itf_mac, unsigned char *almac)
{
	struct topology_discovery_db *tpg_db = NULL;
    struct topology_response_db *tpgr_db = NULL;
    struct device_info_db *dev_db = NULL;

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		SLIST_FOREACH(dev_db, &tpgr_db->devinfo_head, devinfo_entry) {
			if (!os_memcmp(dev_db->mac_addr, itf_mac, ETH_ALEN))
				break;
		}
		if (dev_db)
			break;
	}

	if (!tpgr_db) {
		debug(DEBUG_ERROR, "cannot find tpgr_db by itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)",
			PRINT_MAC(itf_mac));
		return;
	}

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry) {
		if (!os_memcmp(tpg_db->al_mac, tpgr_db->al_mac_addr, ETH_ALEN)) {
			os_memcpy(almac, tpg_db->al_mac, ETH_ALEN);
			debug(DEBUG_OFF, RED("neighbor(%02x:%02x:%02x:%02x:%02x:%02x) find based on"
                  " sta(%02x:%02x:%02x:%02x:%02x:%02x)\n"),PRINT_MAC(tpg_db->al_mac),
                  PRINT_MAC(itf_mac));
			break;
		}
	}
}

struct topology_response_db * find_response_by_almac(
	struct p1905_managerd_ctx *ctx, unsigned char *al_mac)
{
    struct topology_response_db *tpgr_db = NULL;

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry)
	{
		if (!os_memcmp(tpgr_db->al_mac_addr, al_mac, ETH_ALEN)) {
			return tpgr_db;
		}
	}

	return tpgr_db;
}

int find_almac_by_connect_mac(struct p1905_managerd_ctx *ctx,
	unsigned char *mac, unsigned char *almac)
{
	struct topology_discovery_db *tpg_db = NULL;
    struct topology_response_db *tpgr_db = NULL;
	struct device_info_db *dev_info = NULL;

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry) {
		if (!os_memcmp(tpg_db->itf_mac, mac, ETH_ALEN)) {
			os_memcpy(almac, tpg_db->al_mac, ETH_ALEN);
			return 1;
		}
	}

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		SLIST_FOREACH(dev_info, &tpgr_db->devinfo_head, devinfo_entry) {
			if (!os_memcmp(dev_info->mac_addr, mac, ETH_ALEN)) {
				os_memcpy(almac, tpgr_db->al_mac_addr, ETH_ALEN);
				return 2;
			}
		}
	}

	return 0;
}

/*used for PON with MAP device feature
*need reform mesh topology to tree topology in agent side
*/
void mask_control_conn_port(struct p1905_managerd_ctx *ctx,
	struct topology_response_db *tpgr_db)
{
	struct topology_discovery_db *tpg_db = NULL;
	int rsp_port = ETH_ERROR_NOT_SUPPORT, disc_port = ETH_ERROR_NOT_SUPPORT;
	unsigned char role = UNKNOWN;

	if (ctx->role == CONTROLLER)
		return;

	role = (tpgr_db->support_service == 0 || tpgr_db->support_service == 2) ? \
		CONTROLLER : AGENT;
	rsp_port = tpgr_db->eth_port;

	if (role == CONTROLLER && rsp_port >= 0) {
		tpg_db = find_discovery_by_almac(ctx, tpgr_db->al_mac_addr);
		if (!tpg_db) {
			debug(DEBUG_TRACE,
				RED("cannot find discovery from Controller via port %d\n"), rsp_port);
			return;
		}
		disc_port = tpg_db->eth_port;
		if (disc_port == rsp_port) {
			/*receive response & discovery from same eth port, need mark this port
			*as controller-connect port. need drop all agents' MC message.
			*/
			debug(DEBUG_TRACE, RED("recv topology response & discovery"
				" from the same eth_port=%d\n"), rsp_port);
			/*check if need set controller-conncet port*/
			if (!ctx->conn_port.is_set) {
				debug(DEBUG_OFF, RED("set conn_port=%d\n"), rsp_port);
				ctx->conn_port.is_set = 1;
				ctx->conn_port.port_num = rsp_port;
				os_memcpy(ctx->conn_port.filter_almac, tpgr_db->al_mac_addr, ETH_ALEN);
				/*delete existing neighbor info except for controller*/
				delete_topo_disc_db_by_port(ctx, ctx->conn_port.port_num,
					ctx->conn_port.filter_almac);
			}
		} else {
			debug(DEBUG_ERROR, RED("port different!!! recv topology response from %d "
				"receive discovery from the port=%d\n"), rsp_port, disc_port);
		}
	}
}

void unmask_control_conn_port(struct p1905_managerd_ctx *ctx, int port)
{
	if (ctx->conn_port.is_set && (port == ctx->conn_port.port_num)) {
		ctx->conn_port.is_set = 0;
		debug(DEBUG_OFF, RED("clear conn_port=%d\n"), port);
	}
}


